;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

(defpackage #:5am-buchberger
  (:use :cl :it.bese.fiveam :monom :polynomial :infix :symbolic-polynomial :division :priority-queue :buchberger :grobner-debug))

(in-package :5am-buchberger)

(def-suite buchberger-suite 
    :description "Buchberger algorithm suite")

(in-suite buchberger-suite)

(def-fixture buchberger-context ()
  (let ((fl (cdr (string->poly "[x+y,x-2*y]" '(x y))))
	(gb (cdr (string->poly "[x+y,x-2*y,y]" '(x y)))))
    (&body)))


(test buchberger
  "Buchberger algorithm"
  (with-fixture buchberger-context ()
    (is-true (grobner-test gb fl))
    (is (every #'universal-equalp (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; poly_grobner([x-3*u-3*u*v^2+u^3,y-3*v-3*u^2*v+v^3,z-3*u^2+3*v^2],[u,v,x,y,z]);
(def-fixture buchberger-enneper-surface-context ()
  (let ((fl (cdr (string->poly "[x-3*u-3*u*v^2+u^3,y-3*v-3*u^2*v+v^3,z-3*u^2+3*v^2]" '(u v x y z))))
	(gb (cdr (string->poly "[x-3*u*v^2+u^3-3*u,y+v^3-3*u^2*v-3*v,z+3*v^2-3*u^2,
       (-u*z)-3*x+6*u*v^2+9*u,(-v*z)+y-2*v^3-3*v,z^2+6*v^2*z-9*z-9*v*y+9*u*x,
       4*u*v*z-3*u*y+3*v*x,2*u*z^2+6*x*z-18*u*z-9*u*v*y+9*v^2*x,
       (-8*u*z^3)-24*x*z^2+72*u*z^2-36*v^2*x*z+27*u*y^2-27*v*x*y,
       z^3+18*v^2*z^2-18*z^2-54*v*y*z+54*v^2*z+81*z+27*y^2-27*x^2,
       (-4*z^4)+48*z^3-108*v*y*z^2+108*z^2+135*y^2*z+324*v*y*z+108*x^2*z
               -1296*v^2*z-1944*z-243*v^2*y^2-648*y^2+243*v^2*x^2+648*x^2,
       8*v*z^3-9*y*z^2+72*v*z^2+54*v^2*y*z-27*y*z-27*v*y^2+27*v*x^2,
       (-8*v*z^4)+12*y*z^3-96*v*z^3-216*v*z^2-135*v*y^2*z+324*y*z-27*v*x^2*z
                 +81*y^3+81*v*y^2-81*x^2*y-81*v*x^2,
       (-64*v*z^6)+120*y*z^5-1152*v*z^5+288*y*z^4-5184*v*z^4-648*v*y^2*z^3
                  -216*y*z^3+6912*v*z^3+81*y^3*z^2-9720*v*y^2*z^2
                  -1539*x^2*y*z^2+31104*y*z^2+62208*v*z^2+8505*y^3*z
                  +46656*v*y^2*z-8505*x^2*y*z-93312*y*z+729*v*y^4-23328*y^3
                  -1458*v*x^2*y^2-23328*v*y^2+23328*x^2*y+729*v*x^4
                  +23328*v*x^2,
       8*z^6-72*z^5+648*v*y*z^4-648*z^4-945*y^2*z^3+5184*v*y*z^3-189*x^2*z^3
            +5832*z^3+972*y^2*z^2+17496*v*y*z^2-2430*x^2*z^2+8748*v*y^3*z
            -19683*y^2*z+2187*x^2*z-5103*y^4-4374*v*y^3+5832*x^2*y^2
            +4374*v*x^2*y-729*x^4,
       8*z^7-48*z^6+648*v*y*z^5-864*z^5-945*y^2*z^4+5832*v*y*z^4-189*x^2*z^4
            +3888*z^4+81*y^2*z^3+17496*v*y*z^3-2997*x^2*z^3+17496*z^3
            +8748*v*y^3*z^2-16767*y^2*z^2+17496*v*y*z^2-5103*x^2*z^2
            -5103*y^4*z+5832*x^2*y^2*z-6561*y^2*z-729*x^4*z+6561*x^2*z
            -2187*y^4+4374*x^2*y^2-2187*x^4,
       64*z^9-10368*z^7+1296*y^2*z^6-1296*x^2*z^6-34992*y^2*z^5-34992*x^2*z^5
             +419904*z^5+174960*y^2*z^4-174960*x^2*z^4-10935*y^4*z^3
             -56862*x^2*y^2*z^3+314928*y^2*z^3-10935*x^4*z^3+314928*x^2*z^3
             +118098*y^4*z^2-118098*x^4*z^2+59049*y^4*z-118098*x^2*y^2*z
             +59049*x^4*z+19683*y^6-59049*x^2*y^4+59049*x^4*y^2-19683*x^6]" '(u v x y z)))))
    (&body)))
    

(test buchberger-enneper-surface
  "Buchberger algorithm - Enneper surface"
  (with-fixture buchberger-enneper-surface-context ()
    (is-true (grobner-test gb fl))
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; Cyclic roots of degree 5
;; poly_reduced_grobner([x+y+z+u+v,x*y+y*z+z*u+u*v+v*x,x*y*z+y*z*u+z*u*v+u*v*x+v*x*y,x*y*z*u+y*z*u*v+z*u*v*x+u*v*x*y+v*x*y*z,x*y*z*u*v-1],[u,v,x,y,z]);
(def-fixture buchberger-cyclic-5-context ()
  (let ((fl 
	 (cdr (string->poly 
	       "[x+y+z+u+v,x*y+y*z+z*u+u*v+v*x,x*y*z+y*z*u+z*u*v+u*v*x+v*x*y,x*y*z*u+y*z*u*v+z*u*v*x+u*v*x*y+v*x*y*z,x*y*z*u*v-1]"
	       '(u v x y z))))
	(gb (cdr (string->poly "[z+y+x+v+u,
       8*z^12+2*y*z^11+979*z^7+231*y*z^6-55*y^2*z^5-987*z^2-233*y*z+55*y^2,
       z^15+122*z^10-122*z^5-1,
       1467*z^12+566*y*z^11+178981*z^7+69003*y*z^6+550*y^3*z^4-275*y^4*z^3
                -1650*y^5*z^2-179073*z^2-550*y^6*z-69019*y*z-825*v*z-275*y^2
                -275*v^2,(-z^11)-143*z^6-55*v*z^5+144*z+55*v,
       (-346*z^12)-124*y*z^11-42218*z^7-15092*y*z^6-275*y^3*z^4+275*y^4*z^3
                  +440*y^5*z^2+42124*z^2+110*y^6*z+15106*y*z+275*v*z-275*v*y,
       867*z^12+334*y*z^11+105776*z^7+40722*y*z^6+275*y^3*z^4-275*y^4*z^3
               -1045*y^5*z^2-105873*z^2-330*y^6*z-40726*y*z+550*x*z-275*v*z
               -550*y^2+275*x^2+275*v*x,
       (-568*z^13)-232*y*z^12-69289*z^8-28336*y*z^7-550*y^4*z^4+550*y^5*z^3
                  +69307*z^3+275*y^6*z^2+28018*y*z^2-550*x*z^2+550*y^2*z
                  +550*x^2*z+275*x^3,(-z^11)-143*z^6-55*x*z^5+144*z+55*x,
       1121*z^12+442*y*z^11+136763*z^7+53911*y*z^6+275*y^3*z^4-1210*y^5*z^2
                -136674*z^2-440*y^6*z-53913*y*z+275*x*z-275*y^2-275*x*y,
       1042*z^12+398*y*z^11+127116*z^7+48554*y*z^6-55*y^5*z^2-128103*z^2
                -165*y^6*z-48787*y*z-55*y^7+55*y^2]" '(u v x y z)))))
    (&body)))


(test buchberger-cyclic-5
  "Buchberger algorithm - cyclic 5"
  (with-fixture buchberger-cyclic-5-context ()
    (is-true (grobner-test gb fl))
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; A tough example learned from Cox
;; poly_grobner([x^5+y^4+z^3-1,x^3+y^3+z^2-1], [x,y,z]);
(def-fixture buchberger-cox-tough-context ()
  (let ((fl (cdr (string->poly "[x^5+y^4+z^3-1,x^3+y^3+z^2-1]" '(x y z))))
	(gb (cdr (string->poly "[z^3+y^4+x^5-1,z^2+y^3+x^3-1,z^3-x^2*z^2+y^4-x^2*y^3+x^2-1,
       (-z^4)-x*z^3-2*y^3*z^2+2*z^2-y^6-x*y^4+2*y^3+x-1,
       (-y^2*z^4)+x*z^4-x*y^2*z^3+y*z^3+x^2*z^3-2*y^5*z^2+2*x*y^3*z^2
                 +2*y^2*z^2-x^2*y*z^2-2*x*z^2-y^8+3*y^5-2*x*y^3+x*y^2-y^2
                 +x^2*y-y-x^2+x,
       z^7+y^2*z^6+2*y*z^6+3*y^3*z^5+x*y^2*z^5+x*y*z^5-y*z^5-x^2*z^5-3*z^5
          +2*y^5*z^4+5*y^4*z^4-x*y^3*z^4-x*y^2*z^4-2*y^2*z^4-6*y*z^4-x^2*z^4
          -x*z^4+z^4+3*y^6*z^3-9*y^3*z^3+x*y^2*z^3-y*z^3-x^2*z^3+4*z^3+y^8*z^2
          +4*y^7*z^2-y^5*z^2-9*y^4*z^2+2*y^3*z^2+x*y^2*z^2-y^2*z^2-2*x*y*z^2
          +7*y*z^2+3*x^2*z^2+x*z^2-3*z^2+y^9*z-5*y^6*z-x*y^3*z+5*y^3*z
          -x^2*y^2*z+y^2*z+x^2*y*z-x*y*z+2*x*z-2*z+y^10+y^8-5*y^7+y^6-3*y^5
          +4*y^4+2*x*y^3-y^3+x^2*y^2-2*x*y^2+y^2-x^2*y+2*x*y-y-2*x+2,
       z^9+y^2*z^8+y*z^8+3*z^8+3*y^3*z^7+x*y^2*z^7+2*y^2*z^7-y*z^7-x^2*z^7
          +2*x*z^7-3*z^7+2*y^5*z^6+3*y^4*z^6-x*y^3*z^6+9*y^3*z^6+x*y^2*z^6
          -2*y^2*z^6-x*y*z^6-7*y*z^6-x^2*z^6-13*z^6+3*y^6*z^5+5*y^5*z^5
          -7*y^3*z^5+x*y^2*z^5-9*y^2*z^5-2*x*y*z^5-x^2*z^5-3*x*z^5+z^5+y^8*z^4
          +3*y^7*z^4+10*y^6*z^4-y^5*z^4-14*y^4*z^4-30*y^3*z^4-3*x*y^2*z^4
          +y^2*z^4+4*x*y*z^4+14*y*z^4+6*x^2*z^4-3*x*z^4+22*z^4+y^9*z^3
          +4*y^8*z^3-4*y^6*z^3-18*y^5*z^3+2*y^4*z^3+5*x*y^3*z^3+4*y^3*z^3
          +14*y^2*z^3+x*y*z^3-y*z^3-x^2*z^3-2*x*z^3+y^10*z^2+5*y^9*z^2+y^8*z^2
          -9*y^7*z^2-23*y^6*z^2+y^5*z^2+13*y^4*z^2-3*x*y^3*z^2+34*y^3*z^2
          -4*x*y^2*z^2-3*y^2*z^2-x*y*z^2-7*y*z^2-2*x^2*z^2+8*x*z^2-13*z^2
          +y^11*z-8*y^8*z+y^7*z+13*y^5*z-7*x*y^3*z+5*x*y^2*z-5*y^2*z+2*x^2*y*z
          -x*y*z-y*z-2*x^2*z+3*x*z-z+y^12-y^10-7*y^9+2*y^8+5*y^7+14*y^6-3*y^5
          -4*y^4+6*x*y^3-13*y^3-x*y^2+y^2-2*x^2*y+2*y+2*x^2-5*x+3,
       (-z^9)-y^2*z^8-y*z^8-5*z^8-3*y^3*z^7-x*y^2*z^7-4*y^2*z^7-y*z^7+x^2*z^7
             -2*x*z^7+z^7-2*y^5*z^6-3*y^4*z^6+x*y^3*z^6-15*y^3*z^6-3*x*y^2*z^6
             -6*y^2*z^6+x*y*z^6+5*y*z^6+3*x^2*z^6+2*x*z^6+17*z^6-3*y^6*z^5
             -9*y^5*z^5-6*y^4*z^5+2*x*y^3*z^5+y^3*z^5-7*x*y^2*z^5+11*y^2*z^5
             +10*y*z^5+5*x^2*z^5+3*x*z^5+3*z^5-y^8*z^4-3*y^7*z^4-16*y^6*z^4
             -17*y^5*z^4+4*y^4*z^4+8*x*y^3*z^4+38*y^3*z^4+x*y^2*z^4+23*y^2*z^4
             -2*x*y*z^4-3*x*z^4-20*z^4-y^9*z^3-6*y^8*z^3-6*y^7*z^3-2*y^6*z^3
             +16*y^5*z^3+16*y^4*z^3-3*x*y^3*z^3+10*y^3*z^3+4*x*y^2*z^3
             -6*y^2*z^3-x*y*z^3-11*y*z^3-5*x^2*z^3-6*z^3-y^10*z^2-7*y^9*z^2
             -13*y^8*z^2+y^7*z^2+25*y^6*z^2+33*y^5*z^2+y^4*z^2-11*x*y^3*z^2
             -28*y^3*z^2+8*x*y^2*z^2-19*y^2*z^2+3*x*y*z^2-3*y*z^2-4*x^2*z^2
             +11*z^2-y^11*z-2*y^10*z-2*y^9*z+4*y^8*z+9*y^7*z+8*y^6*z-y^5*z
             -8*y^4*z+x*y^3*z-8*y^3*z+x*y^2*z-y^2*z-x*y*z+y*z-x*z+z-y^12
             -2*y^11-y^10+5*y^9+10*y^8+3*y^7-6*y^6-11*y^5-4*y^4+2*x*y^3+5*y^3
             -3*x*y^2+3*y^2+x-1,
       (-y*z^7)+3*y^2*z^6-x*y*z^6-x*z^6-z^6-2*y^4*z^5+3*x*y^2*z^5+2*y*z^5
               +7*y^5*z^4-3*x*y^3*z^4-9*y^2*z^4+y*z^4+3*x*z^4-y^7*z^3
               -3*x*y^2*z^3+2*x*y*z^3-y*z^3+2*z^3+5*y^8*z^2-14*y^5*z^2
               +2*y^4*z^2+6*x*y^3*z^2-3*x*y^2*z^2+9*y^2*z^2-2*y*z^2-3*x*z^2
               +y^11-6*y^8+y^7+7*y^5-3*x*y^3+3*x*y^2-3*y^2-x*y+y+x-1,
       y*z^9-9*z^9-3*y^2*z^8+x*y*z^8+3*y*z^8-8*x*z^8+19*z^8+2*y^4*z^7
            -21*y^3*z^7-3*x*y^2*z^7+12*y^2*z^7+3*x*y*z^7+12*x*z^7+21*z^7
            -7*y^5*z^6+15*y^4*z^6+54*y^3*z^6+9*x*y^2*z^6-4*x*y*z^6-31*y*z^6
            +8*x*z^6-61*z^6+y^7*z^5-15*y^6*z^5+30*y^5*z^5+4*y^4*z^5+42*y^3*z^5
            -3*x*y^2*z^5-42*y^2*z^5-20*x*y*z^5+3*y*z^5-29*z^5-5*y^8*z^4
            +24*y^7*z^4+60*y^6*z^4-83*y^4*z^4-141*y^3*z^4-6*x*y^2*z^4
            +15*y^2*z^4+21*x*y*z^4+48*y*z^4-21*x*z^4+93*z^4-3*y^9*z^3
            +24*y^8*z^3+2*y^7*z^3+9*y^6*z^3-84*y^5*z^3+12*y^4*z^3+24*x*y^3*z^3
            -21*y^3*z^3+54*y^2*z^3+5*x*y*z^3+2*y*z^3-18*x*z^3+5*z^3-y^11*z^2
            +15*y^10*z^2+30*y^9*z^2-4*y^8*z^2-61*y^7*z^2-105*y^6*z^2
            +33*y^5*z^2+77*y^4*z^2-27*x*y^3*z^2+120*y^3*z^2-6*x*y^2*z^2
            -27*y^2*z^2+x*y*z^2-33*y*z^2+32*x*z^2-44*z^2+6*y^11*z-36*y^8*z
            +6*y^7*z+42*y^5*z-18*x*y^3*z+18*x*y^2*z-18*y^2*z-6*x*y*z+6*y*z
            +6*x*z-6*z+3*y^13+6*y^12-2*y^11-18*y^10-33*y^9+18*y^8+28*y^7
            +51*y^6-14*y^5-27*y^4+21*x*y^3-33*y^3-9*x*y^2+9*y^2-x*y+y-11*x+11,
       (-y*z^9)+9*z^9+3*y^2*z^8-x*y*z^8-11*y*z^8+8*x*z^8+5*z^8-2*y^4*z^7
               +21*y^3*z^7+3*x*y^2*z^7+4*y^2*z^7-11*x*y*z^7+4*x*z^7-29*z^7
               +7*y^5*z^6-31*y^4*z^6+18*y^3*z^6+7*x*y^2*z^6-4*x*y*z^6+39*y*z^6
               -8*x*z^6-35*z^6-y^7*z^5+15*y^6*z^5+10*y^5*z^5-4*y^4*z^5
               -42*y^3*z^5+3*x*y^2*z^5-14*y^2*z^5+20*x*y*z^5+5*y*z^5-24*x*z^5
               +29*z^5+5*y^8*z^4-32*y^7*z^4+20*y^6*z^4+83*y^4*z^4-75*y^3*z^4
               -18*x*y^2*z^4-7*y^2*z^4+19*x*y*z^4-56*y*z^4-3*x*z^4+67*z^4
               +3*y^9*z^3+8*y^8*z^3-2*y^7*z^3-9*y^6*z^3-28*y^5*z^3+4*y^4*z^3
               +21*y^3*z^3-8*x*y^2*z^3+10*y^2*z^3-5*x*y*z^3-2*y*z^3+18*x*z^3
               -5*z^3+y^11*z^2-15*y^10*z^2+10*y^9*z^2+4*y^8*z^2+61*y^7*z^2
               -55*y^6*z^2-17*y^5*z^2-77*y^4*z^2+3*x*y^3*z^2+96*y^3*z^2
               +6*x*y^2*z^2+11*y^2*z^2-25*x*y*z^2+33*y*z^2+16*x*z^2-52*z^2
               +2*y^11*z-12*y^8*z+2*y^7*z+14*y^5*z-6*x*y^3*z+6*x*y^2*z-6*y^2*z
               -2*x*y*z+2*y*z+2*x*z-2*z-3*y^13+2*y^12+2*y^11+18*y^10-15*y^9
               -10*y^8-28*y^7+29*y^6+14*y^5+27*y^4+3*x*y^3-39*y^3+x*y^2-y^2
               +9*x*y-9*y-13*x+13,
       y*z^10-6*z^10-3*y^2*z^9+x*y*z^9-y*z^9-5*x*z^9+z^9+2*y^4*z^8-15*y^3*z^8
             -3*x*y^2*z^8+9*y^2*z^8-x*y*z^8+5*y*z^8-x*z^8+32*z^8-7*y^5*z^7
             +4*y^4*z^7+6*y^3*z^7+6*x*y^2*z^7+8*y^2*z^7+4*x*y*z^7-15*y*z^7
             +16*x*z^7-8*z^7+y^7*z^6-12*y^6*z^6+28*y^5*z^6+23*y^4*z^6
             +69*y^3*z^6+2*x*y^2*z^6-30*y^2*z^6-19*x*y*z^6-25*y*z^6-x*z^6
             -69*z^6-5*y^8*z^5+14*y^7*z^5+15*y^6*z^5+20*y^5*z^5-49*y^4*z^5
             -30*y^3*z^5+3*x*y^2*z^5-16*y^2*z^5+2*x*y*z^5+38*y*z^5-6*x*z^5
             +14*z^5-3*y^9*z^4+32*y^8*z^4+28*y^7*z^4+49*y^6*z^4-67*y^5*z^4
             -58*y^4*z^4-108*y^3*z^4-3*x*y^2*z^4+34*y^2*z^4+16*x*y*z^4
             +33*y*z^4-12*x*z^4+61*z^4-y^11*z^3+12*y^10*z^3+12*y^9*z^3
             +12*y^8*z^3-56*y^7*z^3-42*y^6*z^3-29*y^5*z^3+64*y^4*z^3
             +36*y^3*z^3-13*x*y^2*z^3+17*y^2*z^3+9*x*y*z^3-28*y*z^3+5*x*z^3
             +3*z^3+16*y^11*z^2+15*y^10*z^2+20*y^9*z^2-46*y^8*z^2-44*y^7*z^2
             -65*y^6*z^2+40*y^5*z^2+47*y^4*z^2+72*y^3*z^2-11*y^2*z^2-4*x*y*z^2
             -16*y*z^2+4*x*z^2-28*z^2+3*y^13*z+3*y^12*z+2*y^11*z-18*y^10*z
             -15*y^9*z-9*y^8*z+32*y^7*z+21*y^6*z+14*y^5*z-27*y^4*z-6*y^3*z
             +6*x*y^2*z-6*y^2*z-8*x*y*z+8*y*z+2*x*z-2*z+3*y^14+3*y^13+4*y^12
             -14*y^11-15*y^10-21*y^9+10*y^8+25*y^7+34*y^6+y^5-6*y^4-24*y^3
             +2*x*y^2-2*y^2-2*x+2,
       3*z^10-13*y*z^9+3*x*z^9+15*z^9+6*y^3*z^8+16*y^2*z^8-13*x*y*z^8-5*y*z^8
             +11*x*z^8+5*z^8-29*y^4*z^7+45*y^3*z^7+16*x*y^2*z^7+10*y^2*z^7
             -10*x*y*z^7+24*y*z^7+10*x*z^7-71*z^7+3*y^6*z^6+45*y^5*z^6
             -10*y^4*z^6+36*y^3*z^6+10*x*y^2*z^6-44*y^2*z^6-2*x*y*z^6+18*y*z^6
             -33*x*z^6-63*z^6-19*y^7*z^5+50*y^6*z^5+25*y^5*z^5+30*y^4*z^5
             -147*y^3*z^5-12*x*y^2*z^5-30*y^2*z^5+41*x*y*z^5-14*y*z^5-30*x*z^5
             +110*z^5+45*y^8*z^4-5*y^7*z^4+50*y^6*z^4-95*y^5*z^4+29*y^4*z^4
             -135*y^3*z^4-26*x*y^2*z^4+36*y^2*z^4+25*x*y*z^4-24*y*z^4+6*x*z^4
             +103*z^4-3*y^10*z^3+25*y^9*z^3+20*y^8*z^3+3*y^7*z^3-106*y^6*z^3
             -60*y^5*z^3-11*y^4*z^3+153*y^3*z^3-14*x*y^2*z^3+30*y^2*z^3
             -21*x*y*z^3+3*y*z^3+39*x*z^3-54*z^3+19*y^11*z^2+25*y^9*z^2
             -54*y^8*z^2+19*y^7*z^2-100*y^6*z^2+35*y^5*z^2-30*y^4*z^2
             +141*y^3*z^2+12*x*y^2*z^2-4*y^2*z^2-28*x*y*z^2+19*y*z^2+16*x*z^2
             -70*z^2+5*y^12*z+5*y^11*z-30*y^9*z-25*y^8*z+5*y^7*z+50*y^6*z
             +35*y^5*z-45*y^3*z+10*x*y^2*z-10*y^2*z-10*x*z+10*z+3*y^14+5*y^12
             -12*y^11+3*y^10-30*y^9-y^8-3*y^7+53*y^6+15*y^5+21*y^4-54*y^3
             +4*x*y^2-4*y^2+8*x*y-8*y-12*x+12,
       12*z^11-60*y*z^10+12*x*z^10-219*z^10+24*y^3*z^9-60*x*y*z^9+325*y*z^9
              -279*x*z^9-507*z^9-132*y^4*z^8-246*y^3*z^8+240*y^2*z^8
              +217*x*y*z^8+745*y*z^8-395*x*z^8+619*z^8+12*y^6*z^7+60*y^5*z^7
              +1229*y^4*z^7-825*y^3*z^7-386*y^2*z^7+634*x*y*z^7-1744*y*z^7
              +298*x*z^7+1399*z^7-84*y^7*z^6+381*y^6*z^6+875*y^5*z^6
              +2174*y^4*z^6-108*y^3*z^6-386*x*y^2*z^6-1260*y^2*z^6-682*x*y*z^6
              -2642*y*z^6+697*x*z^6+319*z^6+132*y^8*z^5+1723*y^7*z^5
              -210*y^6*z^5-641*y^5*z^5-4450*y^4*z^5+1803*y^3*z^5+180*x*y^2*z^5
              +358*y^2*z^5-1073*x*y*z^5+3238*y*z^5+1038*x*z^5-2018*z^5
              -12*y^10*z^4+660*y^9*z^4+1171*y^8*z^4+2413*y^7*z^4-1834*y^6*z^4
              -2933*y^5*z^4-5633*y^4*z^4+2151*y^3*z^4+546*x*y^2*z^4
              +2180*y^2*z^4-137*x*y*z^4+3356*y*z^4-474*x*z^4-1655*z^4
              +84*y^11*z^3+1059*y^10*z^3+255*y^9*z^3-448*y^8*z^3-4331*y^7*z^3
              -342*y^6*z^3+368*y^5*z^3+5587*y^4*z^3-1497*y^3*z^3-194*x*y^2*z^3
              +202*y^2*z^3+1033*x*y*z^3-2267*y*z^3-975*x*z^3+1330*z^3
              +288*y^12*z^2+677*y^11*z^2+1200*y^10*z^2-1013*y^9*z^2
              -2354*y^8*z^2-4123*y^7*z^2+1940*y^6*z^2+3221*y^5*z^2
              +4582*y^4*z^2-2853*y^3*z^2-180*x*y^2*z^2-1300*y^2*z^2
              +940*x*y*z^2-1823*y*z^2-760*x*z^2+1558*z^2+12*y^14*z+240*y^13*z
              +99*y^12*z-49*y^11*z-1428*y^10*z-354*y^9*z+81*y^8*z+2387*y^7*z
              +282*y^6*z+53*y^5*z-2076*y^4*z+753*y^3*z+174*x*y^2*z-174*y^2*z
              -608*x*y*z+608*y*z+434*x*z-434*z+36*y^15+141*y^14+240*y^13
              -145*y^12-808*y^11-1299*y^10+174*y^9+1361*y^8+2015*y^7-229*y^6
              -1003*y^5-1281*y^4+798*y^3-140*x*y^2+140*y^2-264*x*y+264*y+404*x
              -404,
       75*z^11-761*y*z^10+75*x*z^10-6435*z^10+150*y^3*z^9-761*x*y*z^9
              +10089*y*z^9-7196*x*z^9+1222*z^9-1597*y^4*z^8-10512*y^3*z^8
              +3816*y^2*z^8+8256*x*y*z^8+8082*y*z^8-1359*x*z^8+20901*z^8
              +75*y^6*z^7+761*y^5*z^7+32337*y^4*z^7+13806*y^3*z^7+3866*x*y*z^7
              -51430*y*z^7+10451*x*z^7-10315*z^7-911*y^7*z^6+789*y^6*z^6
              +8991*y^5*z^6+28159*y^4*z^6+36960*y^3*z^6-8454*y^2*z^6
              -24431*x*y*z^6-43050*y*z^6+11835*x*z^6-36365*z^6+1597*y^8*z^5
              +38223*y^7*z^5+26190*y^6*z^5+5499*y^5*z^5-116303*y^4*z^5
              -39249*y^3*z^5+14442*x*y^2*z^5-12720*y^2*z^5-18045*x*y*z^5
              +81708*y*z^5-750*x*z^5+17013*z^5-75*y^10*z^4+7599*y^9*z^4
              +5823*y^8*z^4+35105*y^7*z^4+7548*y^6*z^4-8826*y^5*z^4
              -101550*y^4*z^4-50391*y^3*z^4+2544*x*y^2*z^4-78*y^2*z^4
              +12509*x*y*z^4+76559*y*z^4-13095*x*z^4+29945*z^4+911*y^11*z^3
              +19791*y^10*z^3+18819*y^9*z^3+7043*y^8*z^3-79999*y^7*z^3
              -58590*y^6*z^3-31089*y^5*z^3+114869*y^4*z^3+37977*y^3*z^3
              -20802*x*y^2*z^3+21624*y^2*z^3+29182*x*y*z^3-47799*y*z^3
              -6721*x*z^3-3557*z^3+2958*y^12*z^2-63*y^11*z^2+19080*y^10*z^2
              +4332*y^9*z^2+7386*y^8*z^2-76383*y^7*z^2-35160*y^6*z^2
              -21219*y^5*z^2+117900*y^4*z^2+31239*y^3*z^2-14442*x*y^2*z^2
              +14070*y^2*z^2+11472*x*y*z^2-55446*y*z^2+2970*x*z^2-8694*z^2
              +75*y^14*z+3816*y^13*z+4527*y^12*z+2830*y^11*z-22821*y^10*z
              -23346*y^9*z-14403*y^8*z+41215*y^7*z+33897*y^6*z+22285*y^5*z
              -33819*y^4*z-14256*y^3*z+8904*x*y^2*z-8904*y^2*z-9976*x*y*z
              +9976*y*z+1072*x*z-1072*z+225*y^15-711*y^14+3816*y^13+2616*y^12
              +6723*y^11-23607*y^10-17730*y^9-17211*y^8+42750*y^7+25251*y^6
              +23598*y^5-39996*y^4-5724*y^3+9354*x*y^2-9354*y^2-12072*x*y
              +12072*y+2718*x-2718,
       33*z^11-46*y*z^10+33*x*z^10+57*z^10+66*y^3*z^9-46*x*y*z^9-86*y*z^9
              +11*x*z^9-40*z^9-125*y^4*z^8+285*y^3*z^8-54*y^2*z^8-26*x*y*z^8
              +90*y*z^8+76*x*z^8-239*z^8+33*y^6*z^7+46*y^5*z^7-310*y^4*z^7
              +9*y^3*z^7-32*x*y*z^7+478*y*z^7-23*x*z^7-206*z^7-112*y^7*z^6
              +636*y^6*z^6-184*y^5*z^6-319*y^4*z^6-1068*y^3*z^6+324*y^2*z^6
              +130*x*y*z^6+314*y*z^6-88*x*z^6+367*z^6+125*y^8*z^5-416*y^7*z^5
              -30*y^6*z^5-180*y^5*z^5+1203*y^4*z^5-228*y^3*z^5+180*y^2*z^5
              +244*x*y*z^5-1020*y*z^5-330*x*z^5+361*z^5-33*y^10*z^4
              +744*y^9*z^4-230*y^8*z^4-440*y^7*z^4-1878*y^6*z^4+738*y^5*z^4
              +1051*y^4*z^4+1224*y^3*z^4-36*x*y^2*z^4-612*y^2*z^4+16*x*y*z^4
              -686*y*z^4+15*x*z^4+178*z^4+112*y^11*z^3-246*y^10*z^3-96*y^9*z^3
              -656*y^8*z^3+1098*y^7*z^3+216*y^6*z^3+474*y^5*z^3-1160*y^4*z^3
              +246*y^3*z^3+90*x*y^2*z^3-306*y^2*z^3-254*x*y*z^3+824*y*z^3
              +220*x*z^3-506*z^3+435*y^12*z^2-124*y^11*z^2-270*y^10*z^2
              -1560*y^9*z^2+554*y^8*z^2+956*y^7*z^2+1860*y^6*z^2-960*y^5*z^2
              -1078*y^4*z^2-408*y^3*z^2+414*y^2*z^2-344*x*y*z^2+444*y*z^2
              +344*x*z^2-263*z^2+33*y^14*z-54*y^13*z-30*y^12*z-199*y^11*z
              +357*y^10*z+126*y^9*z+306*y^8*z-640*y^7*z-105*y^6*z-304*y^5*z
              +717*y^4*z-207*y^3*z-126*x*y^2*z+126*y^2*z+232*x*y*z-232*y*z
              -106*x*z+106*z+99*y^15-24*y^14-54*y^13-582*y^12+166*y^11
              +300*y^10+864*y^9-63*y^8-446*y^7-732*y^6+370*y^5+21*y^4+81*y^3
              +72*x*y^2-72*y^2+80*x*y-80*y-152*x+152,
       (-z^10)+z^9-5*y^3*z^8+5*z^8-10*y^6*z^6+3*y^4*z^6+20*y^3*z^6-13*z^6
              -10*y^9*z^4+30*y^6*z^4-30*y^3*z^4+10*z^4+3*y^8*z^3-6*y^4*z^3
              +3*z^3-5*y^12*z^2+20*y^9*z^2-30*y^6*z^2+20*y^3*z^2-5*z^2-y^15
              +6*y^12-10*y^9-3*y^8+10*y^6+3*y^4-5*y^3,
       (-6*y*z^11)-75*z^11-6*x*y*z^10+104*y*z^10-81*x*z^10-12*z^10-12*y^4*z^9
                  -132*y^3*z^9+36*y^2*z^9+86*x*y*z^9+172*y*z^9-73*x*z^9-7*z^9
                  +6*y^5*z^8+337*y^4*z^8+168*y^3*z^8+90*y^2*z^8+76*x*y*z^8
                  -324*y*z^8-74*x*z^8-92*z^8-6*y^7*z^7-21*y^6*z^7+76*y^5*z^7
                  +668*y^4*z^7-9*y^3*z^7-216*y^2*z^7-14*x*y*z^7-962*y*z^7
                  +163*x*z^7+316*z^7+12*y^8*z^6+398*y^7*z^6+456*y^6*z^6
                  +404*y^5*z^6-466*y^4*z^6-906*y^3*z^6-684*y^2*z^6-332*x*y*z^6
                  -58*y*z^6+218*x*z^6+850*z^6+54*y^9*z^5+23*y^8*z^5
                  +928*y^7*z^5-102*y^6*z^5-246*y^5*z^5-2463*y^4*z^5
                  +492*y^3*z^5+108*y^2*z^5-284*x*y*z^5+1914*y*z^5+426*x*z^5
                  -725*z^5+6*y^11*z^4+201*y^10*z^4+384*y^9*z^4+574*y^8*z^4
                  +112*y^7*z^4-1500*y^6*z^4-1662*y^5*z^4-845*y^4*z^4
                  +2034*y^3*z^4+1296*y^2*z^4+142*x*y*z^4+628*y*z^4-159*x*z^4
                  -1316*z^4+18*y^12*z^3-38*y^11*z^3+564*y^10*z^3+102*y^9*z^3
                  +97*y^8*z^3-2346*y^7*z^3-252*y^6*z^3-132*y^5*z^3
                  +3256*y^4*z^3-582*y^3*z^3-126*x*y^2*z^3+234*y^2*z^3
                  +436*x*y*z^3-1474*y*z^3-398*x*z^3+553*z^3+36*y^13*z^2
                  +102*y^12*z^2+332*y^11*z^2+234*y^10*z^2-606*y^9*z^2
                  -1300*y^8*z^2-1108*y^7*z^2+1482*y^6*z^2+1980*y^5*z^2
                  +1154*y^4*z^2-1956*y^3*z^2+108*x*y^2*z^2-846*y^2*z^2
                  +352*x*y*z^2-450*y*z^2-460*x*z^2+946*z^2-21*y^14*z
                  +126*y^13*z+60*y^12*z+137*y^11*z-777*y^10*z-234*y^9*z
                  -216*y^8*z+1334*y^7*z+201*y^6*z+266*y^5*z-1281*y^4*z
                  +405*y^3*z+162*x*y^2*z-162*y^2*z-392*x*y*z+392*y*z+230*x*z
                  -230*z+66*y^14+90*y^13-6*y^12-410*y^11-474*y^10+126*y^9
                  +738*y^8+688*y^7-264*y^6-692*y^5-348*y^4+486*y^3-144*x*y^2
                  +144*y^2-64*x*y+64*y+208*x-208,
       (-3*z^12)+5*y*z^11-3*x*z^11+6*z^11-6*y^3*z^10+5*x*y*z^10+5*y*z^10
                +2*x*z^10+8*z^10+13*y^4*z^9+21*y^3*z^9+2*x*y*z^9-13*y*z^9
                +2*x*z^9-37*z^9-3*y^6*z^8-5*y^5*z^8+13*y^4*z^8+30*y^3*z^8
                -3*x*y*z^8-23*y*z^8-6*x*z^8-36*z^8+11*y^7*z^7+27*y^6*z^7
                -5*y^5*z^7-21*y^4*z^7-87*y^3*z^7-3*x*y*z^7-2*y*z^7-11*x*z^7
                +93*z^7-13*y^8*z^6+11*y^7*z^6+27*y^6*z^6+4*y^5*z^6-47*y^4*z^6
                -87*y^3*z^6-3*x*y*z^6+34*y*z^6+19*x*z^6+84*z^6+3*y^10*z^5
                +15*y^9*z^5-13*y^8*z^5+6*y^7*z^5-84*y^6*z^5+9*y^5*z^5
                -26*y^4*z^5+156*y^3*z^5-18*x*y*z^5+31*y*z^5+21*x*z^5-99*z^5
                -11*y^11*z^4+3*y^10*z^4+15*y^9*z^4+9*y^8*z^4-16*y^7*z^4
                -84*y^6*z^4+6*y^5*z^4+42*y^4*z^4+138*y^3*z^4-3*x*y*z^4
                -11*y*z^4+x*z^4-93*z^4+3*y^12*z^3-11*y^11*z^3+3*y^10*z^3
                -33*y^9*z^3+22*y^8*z^3-16*y^7*z^3+96*y^6*z^3-3*y^5*z^3
                +55*y^4*z^3-132*y^3*z^3+27*x*y*z^3-29*y*z^3-29*x*z^3+45*z^3
                -3*y^14*z^2+3*y^12*z^2+3*y^11*z^2-3*y^10*z^2-33*y^9*z^2
                +13*y^8*z^2-6*y^7*z^2+87*y^6*z^2-4*y^5*z^2+13*y^4*z^2
                -105*y^3*z^2+12*x*y*z^2-13*y*z^2-12*x*z^2+48*z^2-3*y^14*z
                -3*y^12*z+14*y^11*z-3*y^10*z+18*y^9*z-9*y^8*z+5*y^7*z-33*y^6*z
                -y^5*z-21*y^4*z+36*y^3*z-8*x*y*z+8*y*z+8*x*z-8*z-3*y^14-3*y^12
                +14*y^11-3*y^10+18*y^9-9*y^8+5*y^7-33*y^6-y^5-21*y^4+36*y^3
                -8*x*y+8*y+8*x-8,
       315*z^13+315*x*z^12+774*z^12+630*y^3*z^11-1890*y*z^11+1719*x*z^11
               +9852*z^11-315*y^4*z^10-972*y^3*z^10-3150*y^2*z^10-4225*y*z^10
               +8079*x*z^10+3666*z^10+315*y^6*z^9-10224*y^4*z^9+19497*y^3*z^9
               -2700*y^2*z^9-4621*x*y*z^9+6799*y*z^9+3419*x*z^9-34729*z^9
               -630*y^7*z^8-6156*y^6*z^8-13860*y^5*z^8-23729*y^4*z^8
               +21345*y^3*z^8+15300*y^2*z^8-656*x*y*z^8+25824*y*z^8
               -10787*x*z^8-34142*z^8+315*y^8*z^7-17928*y^7*z^7+10779*y^6*z^7
               -19355*y^5*z^7+10688*y^4*z^7-59724*y^3*z^7+28800*y^2*z^7
               +12394*x*y*z^7+6166*y*z^7-23597*x*z^7+59056*z^7-315*y^10*z^6
               -6300*y^9*z^6-21276*y^8*z^6-40003*y^7*z^6+21309*y^6*z^6
               +29183*y^5*z^6+63701*y^4*z^6-59964*y^3*z^6-8550*y^2*z^6
               +13669*x*y*z^6-24442*y*z^6-2422*x*z^6+58708*z^6+630*y^11*z^5
               -12744*y^10*z^5-1620*y^9*z^5-31936*y^8*z^5+8197*y^7*z^5
               -28218*y^6*z^5+69783*y^5*z^5+39183*y^4*z^5+76677*y^3*z^5
               -42750*y^2*z^5-9161*x*y*z^5-37728*y*z^5+10092*x*z^5-38978*z^5
               -1890*y^12*z^4-13572*y^11*z^4-25719*y^10*z^4+195*y^9*z^4
               +16753*y^8*z^4+60793*y^7*z^4-13668*y^6*z^4+9282*y^5*z^4
               -18056*y^4*z^4+59751*y^3*z^4-21600*y^2*z^4-20831*x*y*z^4
               -6707*y*z^4+21327*x*z^4-44501*z^4+315*y^14*z^3-3150*y^13*z^3
               -1494*y^12*z^3-19442*y^11*z^3-9264*y^10*z^3-14991*y^9*z^3
               +54214*y^8*z^3+69093*y^7*z^3+56187*y^6*z^3-51801*y^5*z^3
               -76880*y^4*z^3-53124*y^3*z^3+13950*y^2*z^3-746*x*y*z^3
               +31337*y*z^3+1807*x*z^3+5050*z^3-3006*y^14*z^2-5850*y^13*z^2
               -3489*y^12*z^2+1451*y^11*z^2+14094*y^10*z^2+3354*y^9*z^2
               +24986*y^8*z^2+5933*y^7*z^2+8574*y^6*z^2-39423*y^5*z^2
               -27604*y^4*z^2-19515*y^3*z^2+1350*x*y^2*z^2+19350*y^2*z^2
               +1024*x*y*z^2+13794*y*z^2-2374*x*z^2+7351*z^2-3531*y^14*z
               -6300*y^13*z-4881*y^12*z+17228*y^11*z+34269*y^10*z+22986*y^9*z
               -16443*y^8*z-56365*y^7*z-33441*y^6*z+4073*y^5*z+31983*y^4*z
               +10422*y^3*z-2700*x*y^2*z+2700*y^2*z+7384*x*y*z-7384*y*z
               -4684*x*z+4684*z-3021*y^14-3600*y^13-2346*y^12+15973*y^11
               +18579*y^10+10476*y^9-19638*y^8-29090*y^7-15681*y^6+12118*y^5
               +11253*y^4+4977*y^3+1350*x*y^2-1350*y^2+1544*x*y-1544*y-2894*x
               +2894,
       75*z^14+75*x*z^13+270*z^13+150*y^3*z^12-450*y*z^12+495*x*z^12+1887*z^12
              -75*y^4*z^11-60*y^3*z^11-750*y^2*z^11-420*y*z^11+1722*x*z^11
              +4701*z^11+75*y^6*z^10-2520*y^4*z^10+3039*y^3*z^10-1500*y^2*z^10
              +1850*y*z^10+3252*x*z^10-5517*z^10-150*y^7*z^9-1380*y^6*z^9
              -3300*y^5*z^9-5562*y^4*z^9+14736*y^3*z^9+3000*y^2*z^9
              -748*x*y*z^9+5537*y*z^9-1378*x*z^9-25852*z^9+75*y^8*z^8
              -4440*y^7*z^8+222*y^6*z^8-9480*y^5*z^8-152*y^4*z^8-1290*y^3*z^8
              +11250*y^2*z^8+2297*x*y*z^8+2537*y*z^8-10081*x*z^8-3496*z^8
              -75*y^10*z^7-1500*y^9*z^7-4980*y^8*z^7-11964*y^7*z^7
              +13977*y^6*z^7+760*y^5*z^7+15094*y^4*z^7-49962*y^3*z^7
              +5250*y^2*z^7+5922*x*y*z^7-7042*y*z^7-9036*x*z^7+51603*z^7
              +150*y^11*z^6-3120*y^10*z^6-2100*y^9*z^6-16263*y^8*z^6
              -5464*y^7*z^6+6267*y^6*z^6+26479*y^5*z^6+14988*y^4*z^6
              -19782*y^3*z^6-14250*y^2*z^6+22*x*y*z^6-8196*y*z^6+6539*x*z^6
              +27604*z^6-450*y^12*z^5-3060*y^11*z^5-8922*y^10*z^5+3090*y^9*z^5
              -7543*y^8*z^5+20411*y^7*z^5-29934*y^6*z^5+22579*y^5*z^5
              -5721*y^4*z^5+71151*y^3*z^5-16500*y^2*z^5-12193*x*y*z^5-64*y*z^5
              +13596*x*z^5-45139*z^5+75*y^14*z^4-750*y^13*z^4-870*y^12*z^4
              -10761*y^11*z^4-8022*y^10*z^4+810*y^9*z^4+20939*y^8*z^4
              +29709*y^7*z^4-12909*y^6*z^4-12709*y^5*z^4-16003*y^4*z^4
              +39588*y^3*z^4+750*y^2*z^4-6153*x*y*z^4+4784*y*z^4+6051*x*z^4
              -35013*z^4-630*y^14*z^3-2250*y^13*z^3-522*y^12*z^3-7496*y^11*z^3
              +2943*y^10*z^3-10383*y^9*z^3+25857*y^8*z^3+10659*y^7*z^3
              +38106*y^6*z^3-24488*y^5*z^3-6915*y^4*z^3-50037*y^3*z^3
              +9000*y^2*z^3+7177*x*y*z^3+1581*y*z^3-7559*x*z^3+14575*z^3
              -2328*y^14*z^2-3000*y^13*z^2-1182*y^12*z^2+6013*y^11*z^2
              +11922*y^10*z^2-2523*y^9*z^2+2468*y^8*z^2-15971*y^7*z^2
              +17562*y^6*z^2-5599*y^5*z^2+5948*y^4*z^2-31245*y^3*z^2
              +3750*y^2*z^2+5612*x*y*z^2-2053*y*z^2-5612*x*z^2+16238*z^2
              -2403*y^14*z-2250*y^13*z-2403*y^12*z+11964*y^11*z+11097*y^10*z
              +12168*y^9*z-11709*y^8*z-17745*y^7*z-19683*y^6*z+4449*y^5*z
              +3429*y^4*z+13086*y^3*z-408*x*y*z+408*y*z+408*x*z-408*z
              -1323*y^14-750*y^13-1323*y^12+6424*y^11+3177*y^10+7188*y^9
              -5469*y^8-5045*y^7-12303*y^6+1309*y^5-2511*y^4+10626*y^3
              -1528*x*y+1528*y+1528*x-1528,
       (-2244*z^15)-2244*x*z^14-15867*z^14-4488*y^3*z^13+13464*y*z^13
                   -22599*x*z^13-84498*z^13+2244*y^4*z^12-13782*y^3*z^12
                   +22440*y^2*z^12+59298*y*z^12-102927*x*z^12-323187*z^12
                   -2244*y^6*z^11+83187*y^4*z^11-84696*y^3*z^11
                   +122766*y^2*z^11-34116*y*z^11-262698*x*z^11-349977*z^11
                   +4488*y^7*z^10+33501*y^6*z^10+98736*y^5*z^10
                   +428112*y^4*z^10-729639*y^3*z^10+66012*y^2*z^10
                   -380166*y*z^10-305436*x*z^10+1310613*z^10-2244*y^8*z^9
                   +148422*y^7*z^9+136668*y^6*z^9+626340*y^5*z^9
                   +523962*y^4*z^9-1585704*y^3*z^9-648144*y^2*z^9-592725*y*z^9
                   +435774*x*z^9+2954892*z^9+2244*y^10*z^8+44880*y^9*z^8
                   +141213*y^8*z^8+819048*y^7*z^8-427818*y^6*z^8
                   +984120*y^5*z^8-494016*y^4*z^8+1494546*y^3*z^8
                   -1325370*y^2*z^8-402297*x*y*z^8+50183*y*z^8+1344105*x*z^8
                   -1019772*z^8-4488*y^11*z^7+101139*y^10*z^7+218604*y^9*z^7
                   +1003752*y^8*z^7+1356684*y^7*z^7-1759845*y^6*z^7
                   -848796*y^5*z^7-1921926*y^4*z^7+6169746*y^3*z^7
                   -118842*y^2*z^7-602218*x*y*z^7+985474*y*z^7+791960*x*z^7
                   -6601051*z^7+13464*y^12*z^6+75978*y^11*z^6+590952*y^10*z^6
                   +125628*y^9*z^6+1972755*y^8*z^6-92508*y^7*z^6
                   +123957*y^6*z^6-3443259*y^5*z^6-1174928*y^4*z^6
                   +314898*y^3*z^6+1973514*y^2*z^6+375958*x*y*z^6+700868*y*z^6
                   -1170899*x*z^6-1892048*z^6-2244*y^14*z^5+22440*y^13*z^5
                   +72762*y^12*z^5+639744*y^11*z^5+1153122*y^10*z^5
                   -412266*y^9*z^5+215019*y^8*z^5-3035391*y^7*z^5
                   +3870810*y^6*z^5-2004815*y^5*z^5+1189301*y^4*z^5
                   -9271623*y^3*z^5+1691052*y^2*z^5+1530885*x*y*z^5
                   -275248*y*z^5-1686960*x*z^5+6178311*z^5+11061*y^14*z^4
                   +145206*y^13*z^4+105966*y^12*z^4+1391025*y^11*z^4
                   +731586*y^10*z^4+159402*y^9*z^4-2988399*y^8*z^4
                   -3332521*y^7*z^4+576429*y^6*z^4+2025629*y^5*z^4
                   +1680783*y^4*z^4-3231720*y^3*z^4-347166*y^2*z^4
                   +437669*x*y*z^4-494876*y*z^4-406695*x*z^4+3616217*z^4
                   +135078*y^14*z^3+323418*y^13*z^3+76146*y^12*z^3
                   +647772*y^11*z^3-675759*y^10*z^3+1301451*y^9*z^3
                   -2857513*y^8*z^3-557447*y^7*z^3-4912386*y^6*z^3
                   +2723980*y^5*z^3+293963*y^4*z^3+6721929*y^3*z^3
                   -1046832*y^2*z^3-1034081*x*y*z^3+27047*y*z^3+1082703*x*z^3
                   -2200847*z^3+327084*y^14*z^2+378864*y^13*z^2
                   +181218*y^12*z^2-995829*y^11*z^2-1556670*y^10*z^2
                   +45651*y^9*z^2+35848*y^8*z^2+2216343*y^7*z^2
                   -1624278*y^6*z^2+466235*y^5*z^2-778472*y^4*z^2
                   +3323181*y^3*z^2-389430*y^2*z^2-624300*x*y*z^2+259181*y*z^2
                   +624300*x*z^2-1888926*z^2+302559*y^14*z+256098*y^13*z
                   +302559*y^12*z-1497308*y^11*z-1234029*y^10*z-1559256*y^9*z
                   +1419873*y^8*z+1971349*y^7*z+2559855*y^6*z-496709*y^5*z
                   -186969*y^4*z-1838022*y^3*z+123896*x*y*z-123896*y*z
                   -123896*x*z+123896*z+150819*y^14+77886*y^13+150819*y^12
                   -729784*y^11-316497*y^10-827028*y^9+608229*y^8+501533*y^7
                   +1425351*y^6-131461*y^5+354759*y^4-1264626*y^3+194488*x*y
                   -194488*y-194488*x+194488,
       (-1206891*z^16)-1206891*x*z^15-3020910*z^15-2413782*y^3*z^14
                      +7241346*y*z^14-6641583*x*z^14-28159668*z^14
                      +1206891*y^4*z^13+3613308*y^3*z^13+12068910*y^2*z^13
                      -1184796*y*z^13-21532815*x*z^13-44333787*z^13
                      -1206891*y^6*z^12+39227640*y^4*z^12-55082985*y^3*z^12
                      +10898844*y^2*z^12-33858600*y*z^12-31610193*x*z^12
                      +59910189*z^12+2413782*y^7*z^11+23530692*y^6*z^11
                      +53103204*y^5*z^11+47580399*y^4*z^11-166993284*y^3*z^11
                      -49150374*y^2*z^11+837888*y*z^11-17009715*x*z^11
                      +204859335*z^11-1206891*y^8*z^10+68800152*y^7*z^10
                      -30492435*y^6*z^10+94299528*y^5*z^10-41002956*y^4*z^10
                      +60601743*y^3*z^10-76871310*y^2*z^10+80036142*y*z^10
                      +44060151*x*z^10-34702161*z^10+1206891*y^10*z^9
                      +24137820*y^9*z^9+81461460*y^8*z^9+119269158*y^7*z^9
                      -166665513*y^6*z^9-54879600*y^5*z^9+55955709*y^4*z^9
                      +436861011*y^3*z^9+11689398*y^2*z^9-118500690*y*z^9
                      +50935020*x*z^9-329772720*z^9-2413782*y^11*z^8
                      +48882768*y^10*z^8+7314996*y^9*z^8+171235122*y^8*z^8
                      +1828140*y^7*z^8+40308024*y^6*z^8-132001404*y^5*z^8
                      +223948545*y^4*z^8+19766982*y^3*z^8-62066148*y^2*z^8
                      -327123062*y*z^8+39916506*x*z^8-33709884*z^8
                      +7241346*y^12*z^7+51888948*y^11*z^7+91057935*y^10*z^7
                      -35586606*y^9*z^7+35607951*y^8*z^7+77999124*y^7*z^7
                      +347078946*y^6*z^7+13512516*y^5*z^7-276420654*y^4*z^7
                      -535798128*y^3*z^7-165247050*y^2*z^7-31333184*x*y*z^7
                      -7090804*y*z^7+38409925*x*z^7+272469727*z^7
                      -1206891*y^14*z^6+12068910*y^13*z^6+6056550*y^12*z^6
                      +114029055*y^11*z^6+70885554*y^10*z^6+77088990*y^9*z^6
                      -26572671*y^8*z^6+175263666*y^7*z^6-35487801*y^6*z^6
                      -278535636*y^5*z^6-809357548*y^4*z^6-38002101*y^3*z^6
                      +183134760*y^2*z^6-106624768*x*y*z^6+500924578*y*z^6
                      +77773379*x*z^6-13687279*z^6+11461782*y^14*z^5
                      +22967754*y^13*z^5+8404680*y^12*z^5+61613832*y^11*z^5
                      +141380202*y^10*z^5+204728880*y^9*z^5+46409139*y^8*z^5
                      -239388858*y^7*z^5-540661500*y^6*z^5-516551848*y^5*z^5
                      -362919584*y^4*z^5+458275800*y^3*z^5+431682852*y^2*z^5
                      +1425024*x*y*z^5+428556106*y*z^5-7169976*x*z^5
                      -176398125*z^5+23780835*y^14*z^4+34161930*y^13*z^4
                      +32283972*y^12*z^4+43797039*y^11*z^4+159721179*y^10*z^4
                      +74054580*y^9*z^4-252090588*y^8*z^4-706461920*y^7*z^4
                      -324069963*y^6*z^4+164891836*y^5*z^4+657966399*y^4*z^4
                      +103355523*y^3*z^4+72919944*y^2*z^4+148442176*x*y*z^4
                      -153516094*y*z^4-168881064*x*z^4+60253336*z^4
                      +26303745*y^14*z^3+60060480*y^13*z^3+61390812*y^12*z^3
                      +41332521*y^11*z^3-28380975*y^10*z^3-169325880*y^9*z^3
                      -440088944*y^8*z^3-521738548*y^7*z^3+172096089*y^6*z^3
                      +642176528*y^5*z^3+859376551*y^4*z^3-252707787*y^3*z^3
                      -241043736*y^2*z^3+128634080*x*y*z^3-384373256*y*z^3
                      -137586072*x*z^3+174922400*z^3+32577456*y^14*z^2
                      +74764854*y^13*z^2+59433432*y^12*z^2-98169612*y^11*z^2
                      -287995578*y^10*z^2-220959588*y^9*z^2-38314282*y^8*z^2
                      +235146936*y^7*z^2+287128704*y^6*z^2+207028108*y^5*z^2
                      +91390640*y^4*z^2-114642660*y^3*z^2-128016090*y^2*z^2
                      -22979808*x*y*z^2-109512278*y*z^2+22979808*x*z^2
                      +10139958*z^2+32912346*y^14*z+63570678*y^13*z
                      +32912346*y^12*z-174781174*y^11*z-348511722*y^10*z
                      -133903398*y^9*z+225878394*y^8*z+559662644*y^7*z
                      +171323772*y^6*z-137360800*y^5*z-341749680*y^4*z
                      +50046594*y^3*z-81755552*x*y*z+81755552*y*z+81755552*x*z
                      -81755552*z+12175230*y^14+25603218*y^13+12175230*y^12
                      -65352146*y^11-141444078*y^10-47448162*y^9+87732126*y^8
                      +227205724*y^7+57117876*y^6-55682432*y^5-145202352*y^4
                      +33119766*y^3-35807968*x*y+35807968*y+35807968*x
                      -35807968,
       (-489581*z^17)-489581*x*z^16-292312*z^16-979162*y^3*z^15+2937486*y*z^15
                     -1761055*x*z^15-9016520*z^15+489581*y^4*z^14
                     +3332024*y^3*z^14+4895810*y^2*z^14-6079424*y*z^14
                     -3528903*x*z^14+4289371*z^14-489581*y^6*z^13
                     +14979742*y^4*z^13-24996583*y^3*z^13-4910176*y^2*z^13
                     -13244240*y*z^13+4539755*x*z^13+61250181*z^13
                     +979162*y^7*z^12+10478470*y^6*z^12+21541564*y^5*z^12
                     -11099399*y^4*z^12-24717550*y^3*z^12-29073746*y^2*z^12
                     +24644912*y*z^12+20791993*x*z^12+46992267*z^12
                     -489581*y^8*z^11+26042836*y^7*z^11-30491773*y^6*z^11
                     -2804864*y^5*z^11-56049142*y^4*z^11+156373857*y^3*z^11
                     +2939730*y^2*z^11+32897090*y*z^11+39324507*x*z^11
                     -161411177*z^11+489581*y^10*z^10+9791620*y^9*z^10
                     +33978358*y^8*z^10-4954054*y^7*z^10-45091205*y^6*z^10
                     -98291456*y^5*z^10+40874963*y^4*z^10+153411875*y^3*z^10
                     +62090974*y^2*z^10-97940834*y*z^10-3753886*x*z^10
                     -148351538*z^10-979162*y^11*z^9+18896390*y^10*z^9
                     -15695324*y^9*z^9+6549510*y^8*z^9-96163328*y^7*z^9
                     +140894262*y^6*z^9-30904644*y^5*z^9+31027969*y^4*z^9
                     -279650288*y^3*z^9-13737424*y^2*z^9-22350622*y*z^9
                     -36957362*x*z^9+147937532*z^9+2937486*y^12*z^8
                     +22915264*y^11*z^8-927607*y^10*z^8-21509594*y^9*z^8
                     -122411251*y^8*z^8+4349668*y^7*z^8+123145950*y^6*z^8
                     +76448228*y^5*z^8-269673792*y^4*z^8-279852036*y^3*z^8
                     +22829234*y^2*z^8+248460456*y*z^8-57747753*x*z^8
                     +168811553*z^8-489581*y^14*z^7+4895810*y^13*z^7
                     -3141938*y^12*z^7+6279137*y^11*z^7-44843844*y^10*z^7
                     +51879350*y^9*z^7-70023643*y^8*z^7-32074234*y^7*z^7
                     -227145739*y^6*z^7-96619332*y^5*z^7-53876304*y^4*z^7
                     +203917101*y^3*z^7+205808644*y^2*z^7+177549062*y*z^7
                     -23170081*x*z^7-7660043*z^7+5582660*y^14*z^6
                     -14366*y^13*z^6-1698748*y^12*z^6-65570634*y^11*z^6
                     -16126134*y^10*z^6+19477004*y^9*z^6-22957273*y^8*z^6
                     -229695410*y^7*z^6-195800086*y^6*z^6+114603008*y^5*z^6
                     +515674056*y^4*z^6+205334782*y^3*z^6-28833076*y^2*z^6
                     +71139072*x*y*z^6-235598774*y*z^6-26046530*x*z^6
                     -1195417*z^6+855777*y^14*z^5-4609062*y^13*z^5
                     +4298988*y^12*z^5-50084135*y^11*z^5-80952111*y^10*z^5
                     -115224628*y^9*z^5-144937438*y^8*z^5-5588564*y^7*z^5
                     +164266747*y^6*z^5+529614100*y^5*z^5+509930809*y^4*z^5
                     -19465211*y^3*z^5-357613056*y^2*z^5+10746624*x*y*z^5
                     -384925726*y*z^5-9664776*x*z^5-34373746*z^5
                     -8065907*y^14*z^4-6636972*y^13*z^4-3405556*y^12*z^4
                     -61044931*y^11*z^4-158119059*y^10*z^4-130981024*y^9*z^4
                     +110802808*y^8*z^4+439860884*y^7*z^4+302161305*y^6*z^4
                     +69013032*y^5*z^4-213214897*y^4*z^4-80318471*y^3*z^4
                     -143191576*y^2*z^4-76418336*x*y*z^4-21593012*y*z^4
                     +87611080*x*z^4-89881392*z^4-11389870*y^14*z^3
                     -26326662*y^13*z^3-25762016*y^12*z^3-92246274*y^11*z^3
                     -73532980*y^10*z^3+165780*y^9*z^3+415004346*y^8*z^3
                     +516394464*y^7*z^3+138620594*y^6*z^3-498594316*y^5*z^3
                     -636660514*y^4*z^3-63496082*y^3*z^3+167512282*y^2*z^3
                     -76106656*x*y*z^3+251907926*y*z^3+81491888*x*z^3
                     -61596678*z^3-22171050*y^14*z^2-43988442*y^13*z^2
                     -38326746*y^12*z^2+36464110*y^11*z^2+130477222*y^10*z^2
                     +115078894*y^9*z^2+120119554*y^8*z^2-24803940*y^7*z^2
                     -101183004*y^6*z^2-230519944*y^5*z^2-184697456*y^4*z^2
                     +3945222*y^3*z^2+111282380*y^2*z^2+4327392*x*y*z^2
                     +109647604*y*z^2-4327392*x*z^2+18675596*z^2
                     -30067226*y^14*z-46856342*y^13*z-30067226*y^12*z
                     +155932502*y^11*z+251070826*y^10*z+133547014*y^9*z
                     -183914362*y^8*z-402832596*y^7*z-190170460*y^6*z
                     +99309056*y^5*z+211236496*y^4*z+32812318*y^3*z
                     +44770976*x*y*z-44770976*y*z-44770976*x*z+44770976*z
                     -14178628*y^14-22256476*y^13-14178628*y^12+73585756*y^11
                     +119360228*y^10+62815292*y^9-87048836*y^8-191514888*y^7
                     -89195480*y^6+47205568*y^5+101057888*y^4+14348204*y^3
                     +21540928*x*y-21540928*y-21540928*x+21540928,
       (-92629*z^18)-92629*x*z^17-356523*z^17-185258*y^3*z^16+555774*y*z^16
                    -634410*x*z^16-2404212*z^16+92629*y^4*z^15+27986*y^3*z^15
                    +926290*y^2*z^15+657074*y*z^15-2269604*x*z^15-6033581*z^15
                    -92629*y^6*z^14+3135393*y^4*z^14-3716191*y^3*z^14
                    +2083166*y^2*z^14-3135608*y*z^14-4165226*x*z^14
                    +2131298*z^14+185258*y^7*z^13+1681315*y^6*z^13
                    +4075676*y^5*z^13+7634791*y^4*z^13-18503719*y^3*z^13
                    -3337434*y^2*z^13-3994704*y*z^13-2522718*x*z^13
                    +27531354*z^13-92629*y^8*z^12+5529754*y^7*z^12
                    +159441*y^6*z^12+12722884*y^5*z^12-582783*y^4*z^12
                    -9283157*y^3*z^12-12649852*y^2*z^12+6842690*y*z^12
                    +6654062*x*z^12+24108320*z^12+92629*y^10*z^11
                    +1852580*y^9*z^11+6127507*y^8*z^11+16122558*y^7*z^11
                    -17183616*y^6*z^11+2488624*y^5*z^11-6312235*y^4*z^11
                    +53501482*y^3*z^11-7556812*y^2*z^11+2069580*y*z^11
                    +16177651*x*z^11-39377477*z^11-185258*y^11*z^10
                    +3876425*y^10*z^10+3054784*y^9*z^10+21626108*y^8*z^10
                    +8311782*y^7*z^10-14183737*y^6*z^10-25814020*y^5*z^10
                    +13405734*y^4*z^10+67509729*y^3*z^10+2581674*y^2*z^10
                    -30106884*y*z^10+9624624*x*z^10-78190974*z^10
                    +555774*y^12*z^9+3733146*y^11*z^9+11969039*y^10*z^9
                    -3357566*y^9*z^9+15862111*y^8*z^9-7108532*y^7*z^9
                    +38392044*y^6*z^9-28124408*y^5*z^9-7895845*y^4*z^9
                    -37345552*y^3*z^9+888602*y^2*z^9-17645906*y*z^9
                    -11784367*x*z^9-18699283*z^9-92629*y^14*z^8
                    +926290*y^13*z^8+1212848*y^12*z^8+14249889*y^11*z^8
                    +11942995*y^10*z^8-276048*y^9*z^8-15006568*y^8*z^8
                    -2607006*y^7*z^8+50104927*y^6*z^8-27948056*y^5*z^8
                    -79911876*y^4*z^8-125108127*y^3*z^8+26323650*y^2*z^8
                    +45939334*y*z^8-22766792*x*z^8+87932460*z^8
                    +755025*y^14*z^7+3009456*y^13*z^7+856110*y^12*z^7
                    +13746805*y^11*z^7+8473554*y^10*z^7+20317454*y^9*z^7
                    -32130130*y^8*z^7-29686872*y^7*z^7-27706147*y^6*z^7
                    -31958156*y^5*z^7-54321744*y^4*z^7-65848063*y^3*z^7
                    +50185776*y^2*z^7+61616676*y*z^7-2684085*x*z^7
                    +112103646*z^7+3078241*y^14*z^6+4303472*y^13*z^6
                    +2369872*y^12*z^6-836065*y^11*z^6+5211879*y^10*z^6
                    +23297288*y^9*z^6-52961449*y^8*z^6-67092074*y^7*z^6
                    -104631939*y^6*z^6+51065028*y^5*z^6+66081065*y^4*z^6
                    +106326259*y^3*z^6-6732524*y^2*z^6-2603328*y*z^6
                    +15554654*x*z^6-13118557*z^6+3923996*y^14*z^5
                    +5774610*y^13*z^5+5610976*y^12*z^5-15897300*y^11*z^5
                    -18990852*y^10*z^5-7730476*y^9*z^5-48273990*y^8*z^5
                    -23048880*y^7*z^5-74012650*y^6*z^5+117014132*y^5*z^5
                    +117500814*y^4*z^5+169685596*y^3*z^5-61680680*y^2*z^5
                    -7234336*x*y*z^5-56821566*y*z^5+7549808*x*z^5
                    -113928218*z^5+3097785*y^14*z^4+5610222*y^13*z^4
                    +6898500*y^12*z^4-36197723*y^11*z^4-42585917*y^10*z^4
                    -48744380*y^9*z^4+38402498*y^8*z^4+64859596*y^7*z^4
                    +72925285*y^6*z^4+18957628*y^5*z^4+30999347*y^4*z^4
                    -17032247*y^3*z^4-25082334*y^2*z^4+2348928*x*y*z^4
                    -31568814*y*z^4-3223416*x*z^4-41071590*z^4+96640*y^14*z^3
                    +1279452*y^13*z^3+3269510*y^12*z^3-32101108*y^11*z^3
                    -32746602*y^10*z^3-50861014*y^9*z^3+89014808*y^8*z^3
                    +88902444*y^7*z^3+140069182*y^6*z^3-74522620*y^5*z^3
                    -57498042*y^4*z^3-146873596*y^3*z^3+20574258*y^2*z^3
                    +9250048*x*y*z^3+10620894*y*z^3-9782192*x*z^3+40775794*z^3
                    -5335552*y^14*z^2-3813220*y^13*z^2-3739120*y^12*z^2
                    +9767304*y^11*z^2+4067548*y^10*z^2+1154200*y^9*z^2
                    +22649148*y^8*z^2+9533328*y^7*z^2+25105888*y^6*z^2
                    -34107880*y^5*z^2-34412096*y^4*z^2-40610728*y^3*z^2
                    +13476220*y^2*z^2+1261888*x*y*z^2+11948260*y*z^2
                    -1261888*x*z^2+24316700*z^2-7043730*y^14*z-5731998*y^13*z
                    -7043730*y^12*z+34781406*y^11*z+27348258*y^10*z
                    +36530382*y^9*z-32595186*y^8*z-43669764*y^7*z
                    -60285036*y^6*z+11026752*y^5*z+2281872*y^4*z
                    +44400774*y^3*z-3497952*x*y*z+3497952*y*z+3497952*x*z
                    -3497952*z-3493460*y^14-2695244*y^13-3493460*y^12
                    +17201228*y^11+12678004*y^10+18265516*y^9-15870868*y^8
                    -20231592*y^7-30342328*y^6+5124416*y^5-197024*y^4
                    +23054812*y^3-2128576*x*y+2128576*y+2128576*x-2128576,
       5425752*z^19+5425752*x*z^18+23464704*z^18+10851504*y^3*z^17
                   -32554512*y*z^17+39741960*x*z^17+153678605*z^17
                   -5425752*y^4*z^16+3523392*y^3*z^16-54257520*y^2*z^16
                   -53976192*y*z^16+153537941*x*z^16+422158096*z^16
                   +5425752*y^6*z^15-186237264*y^4*z^15+222728866*y^3*z^15
                   -147835008*y^2*z^15+147859890*y*z^15+317716663*x*z^15
                   +97008176*z^15-10851504*y^7*z^14-95901840*y^6*z^14
                   -238733088*y^5*z^14-537500357*y^4*z^14+1167570136*y^3*z^14
                   +108275182*y^2*z^14+357584768*y*z^14+284863311*x*z^14
                   -1697595811*z^14+5425752*y^8*z^13-329068512*y^7*z^13
                   -53277019*y^6*z^13-858822912*y^5*z^13-267855214*y^4*z^13
                   +1208310223*y^3*z^13+863220400*y^2*z^13-210597280*y*z^13
                   -346502291*x*z^13-2544223629*z^13-5425752*y^10*z^12
                   -108515040*y^9*z^12-356337936*y^8*z^12-1104313162*y^7*z^12
                   +939671858*y^6*z^12-628642108*y^5*z^12+452095871*y^4*z^12
                   -2727921626*y^3*z^12+968343170*y^2*z^12-458715968*y*z^12
                   -1256890345*x*z^12+1354783653*z^12+10851504*y^11*z^11
                   -229643280*y^10*z^11-230560992*y^9*z^11-1434593155*y^8*z^11
                   -1091287060*y^7*z^11+1491307213*y^6*z^11
                   +1459414496*y^5*z^11-275469794*y^4*z^11-6376807425*y^3*z^11
                   +41856558*y^2*z^11+1509882382*y*z^11-1248836715*x*z^11
                   +6638867465*z^11-32554512*y^12*z^10-213506688*y^11*z^10
                   -812030021*y^10*z^10+53215148*y^9*z^10-1734187630*y^8*z^10
                   +214264102*y^7*z^10-1584958987*y^6*z^10+2952251168*y^5*z^10
                   -154562867*y^4*z^10-607630979*y^3*z^10-493850062*y^2*z^10
                   +2456014754*y*z^10+444416950*x*z^10+4157978906*z^10
                   +5425752*y^14*z^9-54257520*y^13*z^9-86530704*y^12*z^9
                   -932889374*y^11*z^9-1145668550*y^10*z^9+203228300*y^9*z^9
                   +397960674*y^8*z^9+923615600*y^7*z^9-4844050734*y^6*z^9
                   +2604905124*y^5*z^9+4716059327*y^4*z^9+10034730488*y^3*z^9
                   -1484843984*y^2*z^9-2066018354*y*z^9+1882110314*x*z^9
                   -5510764628*z^9-41644320*y^14*z^8-202092528*y^13*z^8
                   -101443374*y^12*z^8-1338827536*y^11*z^8-823634969*y^10*z^8
                   -1054278406*y^9*z^8+3029383819*y^8*z^8+1785652364*y^7*z^8
                   -506947854*y^6*z^8+2195417116*y^5*z^8+7015201440*y^4*z^8
                   +9010490868*y^3*z^8-3809196434*y^2*z^8-6369412632*y*z^8
                   +1135657977*x*z^8-10022492105*z^8-198432619*y^14*z^7
                   -365104946*y^13*z^7-143957614*y^12*z^7-371519033*y^11*z^7
                   -274302996*y^10*z^7-2239865174*y^9*z^7+4414715179*y^8*z^7
                   +4948278298*y^7*z^7+8253957763*y^6*z^7-1525015836*y^5*z^7
                   -2035970664*y^4*z^7-5607716901*y^3*z^7-2230127044*y^2*z^7
                   -2622212534*y*z^7-698300063*x*z^7-2309997637*z^7
                   -348885308*y^14*z^6-458089666*y^13*z^6-384586916*y^12*z^6
                   +1345067346*y^11*z^6+1063206942*y^10*z^6-312441548*y^9*z^6
                   +4440304537*y^8*z^6+4587997010*y^7*z^6+8417442814*y^6*z^6
                   -8959819328*y^5*z^6-11795833296*y^4*z^6-14125505686*y^3*z^6
                   +3972318964*y^2*z^6+4804260374*y*z^6-720549214*x*z^6
                   +7046057497*z^6-295902513*y^14*z^5-462088458*y^13*z^5
                   -586051692*y^12*z^5+2861637239*y^11*z^5+3505906095*y^10*z^5
                   +3756990292*y^9*z^5-40810802*y^8*z^5-3123553132*y^7*z^5
                   -3187541323*y^6*z^5-7526070004*y^5*z^5-8127730537*y^4*z^5
                   -3615097525*y^3*z^5+5318267856*y^2*z^5+5725495246*y*z^5
                   +35987592*x*z^5+5785417954*z^5-43942237*y^14*z^4
                   -191752548*y^13*z^4-363469916*y^12*z^4+3252689203*y^11*z^4
                   +4046760387*y^10*z^4+5117782288*y^9*z^4-6944247016*y^8*z^4
                   -9635047988*y^7*z^4-12036683961*y^6*z^4+3425779992*y^5*z^4
                   +3774129649*y^4*z^4+9556210487*y^3*z^4+346787320*y^2*z^4
                   -152082400*x*y*z^4+386246036*y*z^4+140951096*x*z^4
                   -708490224*z^4+377683006*y^14*z^3+344523798*y^13*z^3
                   +281362064*y^12*z^3+871936914*y^11*z^3+1112320036*y^10*z^3
                   +1348773660*y^9*z^3-6279338058*y^8*z^3-6111893760*y^7*z^3
                   -6199672370*y^6*z^3+7044591244*y^5*z^3+7410393154*y^4*z^3
                   +6849999842*y^3*z^3-2360538778*y^2*z^3+121651360*x*y*z^3
                   -2496380054*y*z^3-138899888*x*z^3-2193760698*z^3
                   +693341130*y^14*z^2+704041722*y^13*z^2+745086714*y^12*z^2
                   -2526718798*y^11*z^2-2492488582*y^10*z^2-2857421902*y^9*z^2
                   +562578782*y^8*z^2+2440039332*y^7*z^2+3434277084*y^6*z^2
                   +1677738760*y^5*z^2+1925499248*y^4*z^2-1492560870*y^3*z^2
                   -1038420620*y^2*z^2+143950368*x*y*z^2-1190995252*y*z^2
                   -143950368*x*z^2-583996748*z^2+580021274*y^14*z
                   +596718230*y^13*z+580021274*y^12*z-2905672022*y^11*z
                   -3000288106*y^10*z-2883409414*y^9*z+2933500282*y^8*z
                   +4801574100*y^7*z+4590079324*y^6*z-1199002112*y^5*z
                   -1310315152*y^4*z-2783227678*y^3*z-44525216*x*y*z
                   +44525216*y*z+44525216*x*z-44525216*z+181811332*y^14
                   +207684124*y^13+181811332*y^12-917680924*y^11
                   -1064293412*y^10-883183868*y^9+960802244*y^8+1704594312*y^7
                   +1376872280*y^6-423992512*y^5-596477792*y^4-727947116*y^3
                   -68994112*x*y+68994112*y+68994112*x-68994112,
       (-1901030*z^20)-1901030*x*z^19-9124762*z^19-3802060*y^3*z^18
                      +11406180*y*z^18-14827852*x*z^18-58433804*z^18
                      +1901030*y^4*z^17-3041284*y^3*z^17+19010300*y^2*z^17
                      +24332092*y*z^17-61094700*x*z^17-176125920*z^17
                      -1901030*y^6*z^16+66155662*y^4*z^16-79989082*y^3*z^16
                      +60831140*y^2*z^16-38725208*y*z^16-141555986*x*z^16
                      -121987211*z^16+3802060*y^7*z^15+32697898*y^6*z^15
                      +83645320*y^5*z^15+220015894*y^4*z^15-445962006*y^3*z^15
                      -6499172*y^2*z^15-145066660*y*z^15-169425253*x*z^15
                      +534196514*z^15-1901030*y^8*z^14+117103084*y^7*z^14
                      +33952250*y^6*z^14+340656568*y^5*z^14+206437392*y^4*z^14
                      -645131456*y^3*z^14-305132132*y^2*z^14-8846618*y*z^14
                      +43295079*x*z^14+1189773846*z^14+1901030*y^10*z^13
                      +38020600*y^9*z^13+123947338*y^8*z^13+443074892*y^7*z^13
                      -307979826*y^6*z^13+393273232*y^5*z^13-57568451*y^4*z^13
                      +618310242*y^3*z^13-507589450*y^2*z^13+166362300*y*z^13
                      +479490441*x*z^13+151723609*z^13-3802060*y^11*z^12
                      +81363902*y^10*z^12+98849920*y^9*z^12+561289236*y^8*z^12
                      +606955512*y^7*z^12-677793853*y^6*z^12
                      -312956032*y^5*z^12+16949814*y^4*z^12
                      +2620078601*y^3*z^12-269069540*y^2*z^12-402242712*y*z^12
                      +695841945*x*z^12-2374071553*z^12+11406180*y^12*z^11
                      +72999916*y^11*z^11+323430470*y^10*z^11
                      +33388972*y^9*z^11+891604480*y^8*z^11+225381030*y^7*z^11
                      +180452542*y^6*z^11-1259049116*y^5*z^11
                      +53526949*y^4*z^11+1668713610*y^3*z^11
                      +110401322*y^2*z^11-1096671392*y*z^11+171379965*x*z^11
                      -2852256305*z^11-1901030*y^14*z^10+19010300*y^13*z^10
                      +35738272*y^12*z^10+361042598*y^11*z^10
                      +565165996*y^10*z^10-57565448*y^9*z^10
                      +308599645*y^8*z^10-298062552*y^7*z^10
                      +1856643697*y^6*z^10-1594374328*y^5*z^10
                      -1527897338*y^4*z^10-2917463441*y^3*z^10
                      +621489522*y^2*z^10+93189406*y*z^10-662542785*x*z^10
                      +662583039*z^10+13687598*y^14*z^9+79841440*y^13*z^9
                      +54043964*y^12*z^9+651911314*y^11*z^9+567492835*y^10*z^9
                      +310821040*y^9*z^9-1010836984*y^8*z^9-831783810*y^7*z^9
                      +1266941069*y^6*z^9-1410085264*y^5*z^9
                      -3301314877*y^4*z^9-5102887683*y^3*z^9
                      +1588407894*y^2*z^9+2445683234*y*z^9-798074994*x*z^9
                      +4291414128*z^9+75776730*y^14*z^8+168393768*y^13*z^8
                      +76262484*y^12*z^8+458045256*y^11*z^8+321211684*y^10*z^8
                      +958290616*y^9*z^8-2161719654*y^8*z^8-2050251984*y^7*z^8
                      -2438492066*y^6*z^8-37071676*y^5*z^8-1043292605*y^4*z^8
                      -456970028*y^3*z^8+1609498744*y^2*z^8+2317634842*y*z^8
                      -112115232*x*z^8+3125790660*z^8+160840014*y^14*z^7
                      +243458536*y^13*z^7+165023254*y^12*z^7
                      -308161482*y^11*z^7-264432909*y^10*z^7+632061294*y^9*z^7
                      -2527470259*y^8*z^7-2650060248*y^7*z^7
                      -4527599648*y^6*z^7+3157792356*y^5*z^7
                      +4071804254*y^4*z^7+5397856666*y^3*z^7-650820266*y^2*z^7
                      -792830676*y*z^7+348957823*x*z^7-1258409243*z^7
                      +184438803*y^14*z^6+269872886*y^13*z^6
                      +286825646*y^12*z^6-1232750031*y^11*z^6
                      -1367006692*y^10*z^6-1092722282*y^9*z^6
                      -1115110457*y^8*z^6-163676530*y^7*z^6-1055368119*y^6*z^6
                      +4504867188*y^5*z^6+5298473826*y^4*z^6
                      +4401698661*y^3*z^6-2574355204*y^2*z^6-2825141214*y*z^6
                      +221932859*x*z^6-3296849257*z^6+93566868*y^14*z^5
                      +186656478*y^13*z^5+266256132*y^12*z^5
                      -1733209434*y^11*z^5-2141486900*y^10*z^5
                      -2475608812*y^9*z^5+2084294867*y^8*z^5
                      +3726153902*y^7*z^5+4202893538*y^6*z^5+914684024*y^5*z^5
                      +896394002*y^4*z^5-1496468530*y^3*z^5-1461318256*y^2*z^5
                      -1507156886*y*z^5+231768*x*z^5-1554195369*z^5
                      -102195903*y^14*z^4-47461622*y^13*z^4+12748556*y^12*z^4
                      -1113696963*y^11*z^4-1377187127*y^10*z^4
                      -1683720988*y^9*z^4+3639185236*y^8*z^4
                      +4223411688*y^7*z^4+4713450611*y^6*z^4
                      -2898986852*y^5*z^4-2996459359*y^4*z^4
                      -4116618567*y^3*z^4+584578600*y^2*z^4+577826674*y*z^4
                      +1455624*x*z^4+584078344*z^4-305099931*y^14*z^3
                      -294615928*y^13*z^3-283822424*y^12*z^3
                      +503667781*y^11*z^3+446895969*y^10*z^3+401964620*y^9*z^3
                      +1504054128*y^8*z^3+817537780*y^7*z^3+860674985*y^6*z^3
                      -2309669544*y^5*z^3-2331987649*y^4*z^3
                      -1699396367*y^3*z^3+908407628*y^2*z^3-2558880*x*y*z^3
                      +911170484*y*z^3+1511208*x*z^3+870218468*z^3
                      -357964740*y^14*z^2-354383622*y^13*z^2
                      -354821724*y^12*z^2+1510708928*y^11*z^2
                      +1496178122*y^10*z^2+1494530312*y^9*z^2
                      -954660462*y^8*z^2-2018386032*y^7*z^2-1995124344*y^6*z^2
                      -110474580*y^5*z^2-114966768*y^4*z^2+924554040*y^3*z^2
                      +272158870*y^2*z^2+927072*x*y*z^2+270707962*y*z^2
                      -927072*x*z^2+291944038*z^2-212157154*y^14*z
                      -214340590*y^13*z-212157154*y^12*z+1061513582*y^11*z
                      +1073886386*y^10*z+1058602334*y^9*z-1065152642*y^8*z
                      -1718363780*y^7*z-1690706924*y^6*z+429408992*y^5*z
                      +443965232*y^4*z+1045501718*y^3*z+5822496*x*y*z
                      -5822496*y*z-5822496*x*z+5822496*z-56003282*y^14
                      -54431774*y^13-56003282*y^12+279492574*y^11
                      +270587362*y^10+281587918*y^9-276873394*y^8
                      -432835012*y^7-452740780*y^6+108339712*y^5+97862992*y^4
                      +291016966*y^3-4190688*x*y+4190688*y+4190688*x-4190688,
       (-44425*z^21)-44425*x*z^20-193795*z^20-88850*y^3*z^19+266550*y*z^19
                    -327070*x*z^19-1274354*z^19+44425*y^4*z^18-32190*y^3*z^18
                    +444250*y^2*z^18+451970*y*z^18-1278214*x*z^18-3527528*z^18
                    -44425*y^6*z^17+1526545*y^4*z^17-1842423*y^3*z^17
                    +1227150*y^2*z^17-1140996*y*z^17-2698855*x*z^17
                    -1109995*z^17+88850*y^7*z^16+783555*y^6*z^16
                    +1954700*y^5*z^16+4467129*y^4*z^16-9605029*y^3*z^16
                    -752630*y^2*z^16-2972806*y*z^16-2572042*x*z^16
                    +13565093*z^16-44425*y^8*z^15+2697690*y^7*z^15
                    +456911*y^6*z^15+7105380*y^5*z^15+2647468*y^4*z^15
                    -10602997*y^3*z^15-7006014*y^2*z^15+1218640*y*z^15
                    +2619449*x*z^15+22302653*z^15+44425*y^10*z^14
                    +888500*y^9*z^14+2915955*y^8*z^14+9160898*y^7*z^14
                    -7506655*y^6*z^14+5800536*y^5*z^14-3245086*y^4*z^14
                    +20587543*y^3*z^14-8783974*y^2*z^14+3837574*y*z^14
                    +10650408*x*z^14-7954033*z^14-88850*y^11*z^13
                    +1881945*y^10*z^13+1921200*y^9*z^13+11847046*y^8*z^13
                    +9782204*y^7*z^13-12668822*y^6*z^13-10997536*y^5*z^13
                    +1090138*y^4*z^13+54430174*y^3*z^13-1436450*y^2*z^13
                    -11018460*y*z^13+11493837*x*z^13-56030677*z^13
                    +266550*y^12*z^12+1744810*y^11*z^12+6728289*y^10*z^12
                    -187950*y^9*z^12+15235952*y^8*z^12-505866*y^7*z^12
                    +10778914*y^6*z^12-25975004*y^5*z^12+899768*y^4*z^12
                    +13274582*y^3*z^12+4950830*y^2*z^12-21334144*y*z^12
                    -2616810*x*z^12-42908546*z^12-44425*y^14*z^11
                    +444250*y^13*z^11+718520*y^12*z^11+7686377*y^11*z^11
                    +9990070*y^10*z^11-1596036*y^9*z^11-1342215*y^8*z^11
                    -8841150*y^7*z^11+40955979*y^6*z^11-24957052*y^5*z^11
                    -36144372*y^4*z^11-82735585*y^3*z^11+13378064*y^2*z^11
                    +12799086*y*z^11-16744470*x*z^11+42041860*z^11
                    +339305*y^14*z^10+1671400*y^13*z^10+910274*y^12*z^10
                    +11626231*y^11*z^10+7801347*y^10*z^10+7831314*y^9*z^10
                    -26096155*y^8*z^10-16474014*y^7*z^10+11243399*y^6*z^10
                    -17808216*y^5*z^10-61462426*y^4*z^10-89174777*y^3*z^10
                    +30957924*y^2*z^10+55234262*y*z^10-12049440*x*z^10
                    +91874913*z^10+1628711*y^14*z^9+3140020*y^13*z^9
                    +1263518*y^12*z^9+4404103*y^11*z^9+2153415*y^10*z^9
                    +19135710*y^9*z^9-40336258*y^8*z^9-39769140*y^7*z^9
                    -68036522*y^6*z^9+12529572*y^5*z^9+7525711*y^4*z^9
                    +37559804*y^3*z^9+21952328*y^2*z^9+29962348*y*z^9
                    +4801407*x*z^9+31327296*z^9+3000105*y^14*z^8
                    +4046756*y^13*z^8+3116408*y^12*z^8-11359143*y^11*z^8
                    -9140507*y^10*z^8+5385232*y^9*z^8-38148698*y^8*z^8
                    -41664378*y^7*z^8-80668677*y^6*z^8+73310048*y^5*z^8
                    +103064435*y^4*z^8+127272309*y^3*z^8-30170622*y^2*z^8
                    -39724286*y*z^8+8854766*x*z^8-57432520*z^8
                    +2743338*y^14*z^7+3960482*y^13*z^7+5071808*y^12*z^7
                    -25510554*y^11*z^7-29133398*y^10*z^7-31118752*y^9*z^7
                    -1947773*y^8*z^7+21330064*y^7*z^7+18392944*y^6*z^7
                    +73580532*y^5*z^7+82980083*y^4*z^7+49867042*y^3*z^7
                    -52627432*y^2*z^7-56881602*y*z^7+1892306*x*z^7
                    -63266461*z^7+437586*y^14*z^6+1782242*y^13*z^6
                    +3440132*y^12*z^6-28425462*y^11*z^6-36482669*y^10*z^6
                    -46554784*y^9*z^6+58365386*y^8*z^6+86946000*y^7*z^6
                    +105699982*y^6*z^6-21170564*y^5*z^6-28598633*y^4*z^6
                    -74430398*y^3*z^6-9384858*y^2*z^6-8218528*y*z^6
                    -1980847*x*z^6-5375344*z^6-3228714*y^14*z^5
                    -2836264*y^13*z^5-2194496*y^12*z^5-9426528*y^11*z^5
                    -11662110*y^10*z^5-15507324*y^9*z^5+63744949*y^8*z^5
                    +61819454*y^7*z^5+68420946*y^6*z^5-74140892*y^5*z^5
                    -75995106*y^4*z^5-79475920*y^3*z^5+26513908*y^2*z^5
                    +26664768*y*z^5+17496*x*z^5+27268337*z^5-6067476*y^14*z^4
                    -6324094*y^13*z^4-6620078*y^12*z^4+21880364*y^11*z^4
                    +22936006*y^10*z^4+24599634*y^9*z^4+662692*y^8*z^4
                    -20296984*y^7*z^4-23355948*y^6*z^4-25675184*y^5*z^4
                    -25336908*y^4*z^4-1372204*y^3*z^4+15113990*y^2*z^4
                    +15232088*y*z^4-34992*x*z^4+14641598*z^4-5393667*y^14*z^3
                    -5404156*y^13*z^3-5499938*y^12*z^3+29810057*y^11*z^3
                    +29956543*y^10*z^3+30284550*y^9*z^3-35221464*y^8*z^3
                    -53124800*y^7*z^3-52987825*y^6*z^3+18268092*y^5*z^3
                    +18247517*y^4*z^3+36291181*y^3*z^3-2001554*y^2*z^3
                    -2010302*y*z^3+17496*x*z^3-1214234*z^3-1569800*y^14*z^2
                    -1661654*y^13*z^2-1622288*y^12*z^2+10350756*y^11*z^2
                    +10775034*y^10*z^2+10578204*y^9*z^2-15349894*y^8*z^2
                    -20474824*y^7*z^2-20457328*y^6*z^2+10504900*y^5*z^2
                    +10793584*y^4*z^2+15564220*y^3*z^2-2374910*y^2*z^2
                    +69984*x*y*z^2-2436146*y*z^2-69984*x*z^2-2619854*z^2
                    +632812*y^14*z+685300*y^13*z+632812*y^12*z-3181556*y^11*z
                    -3478988*y^10*z-3111572*y^9*z+3269036*y^8*z+5569880*y^7*z
                    +4905032*y^6*z-1388096*y^5*z-1738016*y^4*z-2796644*y^3*z
                    -139968*x*y*z+139968*y*z+139968*x*z-139968*z+501226*y^14
                    +474982*y^13+501226*y^12-2497382*y^11-2348666*y^10
                    -2532374*y^9+2453642*y^8+3756116*y^7+4088540*y^6
                    -941216*y^5-766256*y^4-2689838*y^3+69984*x*y-69984*y
                    -69984*x+69984,
       z^22+x*z^21+6*z^21+2*y^3*z^20-6*y*z^20+9*x*z^20+37*z^20-y^4*z^19
           +4*y^3*z^19-10*y^2*z^19-20*y*z^19+42*x*z^19+132*z^19+y^6*z^18
           -36*y^4*z^18+45*y^3*z^18-44*y^2*z^18+2*y*z^18+117*x*z^18+191*z^18
           -2*y^7*z^17-16*y^6*z^17-44*y^5*z^17-158*y^4*z^17+286*y^3*z^17
           -40*y^2*z^17+94*y*z^17+195*x*z^17-156*z^17+y^8*z^16-64*y^7*z^16
           -38*y^6*z^16-232*y^5*z^16-265*y^4*z^16+642*y^3*z^16+148*y^2*z^16
           +106*y*z^16+123*x*z^16-927*z^16-y^10*z^15-20*y^9*z^15-64*y^8*z^15
           -308*y^7*z^15+132*y^6*z^15-444*y^5*z^15-160*y^4*z^15+202*y^3*z^15
           +460*y^2*z^15-42*y*z^15-232*x*z^15-972*z^15+2*y^11*z^14
           -44*y^10*z^14-76*y^9*z^14-373*y^8*z^14-630*y^7*z^14+541*y^6*z^14
           -176*y^5*z^14-32*y^4*z^14-1587*y^3*z^14+544*y^2*z^14+112*y*z^14
           -678*x*z^14+822*z^14-6*y^12*z^13-36*y^11*z^13-222*y^10*z^13
           -90*y^9*z^13-856*y^8*z^13-622*y^7*z^13+414*y^6*z^13+748*y^5*z^13
           -25*y^4*z^13-2686*y^3*z^13+254*y^2*z^13+784*y*z^13-660*x*z^13
           +2938*z^13+y^14*z^12-10*y^13*z^12-26*y^12*z^12-235*y^11*z^12
           -523*y^10*z^12-18*y^9*z^12-877*y^8*z^12-156*y^7*z^12-902*y^6*z^12
           +1712*y^5*z^12+770*y^4*z^12-240*y^3*z^12-318*y^2*z^12+746*y*z^12
           +45*x*z^12+2097*z^12-6*y^14*z^11-54*y^13*z^11-54*y^12*z^11
           -590*y^11*z^11-743*y^10*z^11-138*y^9*z^11+88*y^8*z^11+558*y^7*z^11
           -1874*y^6*z^11+2096*y^5*z^11+2683*y^4*z^11+4030*y^3*z^11
           -1258*y^2*z^11-1040*y*z^11+783*x*z^11-1862*z^11-48*y^14*z^10
           -144*y^13*z^10-84*y^12*z^10-750*y^11*z^10-684*y^10*z^10
           -684*y^9*z^10+1680*y^8*z^10+1686*y^7*z^10-26*y^6*z^10+1356*y^5*z^10
           +3037*y^4*z^10+4238*y^3*z^10-2018*y^2*z^10-2768*y*z^10+744*x*z^10
           -4500*z^10-136*y^14*z^9-256*y^13*z^9-150*y^12*z^9-308*y^11*z^9
           -223*y^10*z^9-1018*y^9*z^9+2968*y^8*z^9+2916*y^7*z^9+3544*y^6*z^9
           -1244*y^5*z^9-565*y^4*z^9-1108*y^3*z^9-1108*y^2*z^9-1716*y*z^9
           +115*x*z^9-2504*z^9-219*y^14*z^8-342*y^13*z^8-276*y^12*z^8
           +711*y^11*z^8+795*y^10*z^8-88*y^9*z^8+2781*y^8*z^8+2318*y^7*z^8
           +4061*y^6*z^8-4336*y^5*z^8-5025*y^4*z^8-5533*y^3*z^8+1306*y^2*z^8
           +1318*y*z^8-300*x*z^8+1617*z^8-210*y^14*z^7-336*y^13*z^7
           -366*y^12*z^7+1770*y^11*z^7+2058*y^10*z^7+1806*y^9*z^7+314*y^8*z^7
           -1112*y^7*z^7-270*y^6*z^7-4180*y^5*z^7-4922*y^4*z^7-3486*y^3*z^7
           +2552*y^2*z^7+2770*y*z^7-240*x*z^7+3228*z^7-57*y^14*z^6
           -168*y^13*z^6-254*y^12*z^6+2019*y^11*z^6+2449*y^10*z^6+2738*y^9*z^6
           -2900*y^8*z^6-4496*y^7*z^6-4787*y^6*z^6-324*y^5*z^6-497*y^4*z^6
           +1823*y^3*z^6+1334*y^2*z^6+1442*y*z^6-64*x*z^6+1614*z^6
           +198*y^14*z^5+132*y^13*z^5+66*y^12*z^5+922*y^11*z^5+1232*y^10*z^5
           +1542*y^9*z^5-3648*y^8*z^5-4116*y^7*z^5-4584*y^6*z^5+2748*y^5*z^5
           +2820*y^4*z^5+3882*y^3*z^5-418*y^2*z^5-398*y*z^5-378*z^5
           +408*y^14*z^4+384*y^13*z^4+360*y^12*z^4-792*y^11*z^4-672*y^10*z^4
           -552*y^9*z^4-1472*y^8*z^4-656*y^7*z^4-848*y^6*z^4+2320*y^5*z^4
           +2368*y^4*z^4+1768*y^3*z^4-872*y^2*z^4-872*y*z^4-872*z^4
           +422*y^14*z^3+418*y^13*z^3+414*y^12*z^3-1642*y^11*z^3-1622*y^10*z^3
           -1602*y^9*z^3+718*y^8*z^3+1916*y^7*z^3+1884*y^6*z^3+512*y^5*z^3
           +520*y^4*z^3-642*y^3*z^3-432*y^2*z^3-432*y*z^3-432*z^3+264*y^14*z^2
           +264*y^13*z^2+264*y^12*z^2-1240*y^11*z^2-1240*y^10*z^2-1240*y^9*z^2
           +1080*y^8*z^2+1872*y^7*z^2+1872*y^6*z^2-288*y^5*z^2-288*y^4*z^2
           -1080*y^3*z^2-80*y^2*z^2-80*y*z^2-80*z^2+96*y^14*z+96*y^13*z
           +96*y^12*z-480*y^11*z-480*y^10*z-480*y^9*z+480*y^8*z+768*y^7*z
           +768*y^6*z-192*y^5*z-192*y^4*z-480*y^3*z+16*y^14+16*y^13+16*y^12
           -80*y^11-80*y^10-80*y^9+80*y^8+128*y^7+128*y^6-32*y^5-32*y^4
           -80*y^3]" '(x y z))))))
  (&body))

(test buchberger-cox-tough
  "Buchberger algorithm - Cox tough example"
  (with-fixture buchberger-cox-tough-context ()
    (is-true (grobner-test gb fl))
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )


(run! 'buchberger-suite)
(format t "All tests done!~%")


