;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "COPY"
  (:use :cl)
  (:export "COPY-INSTANCE")
  (:shadowing-import-from 
   #+openmcl-native-threads #:ccl 
   #+cmu #:pcl
   #+sbcl #:sb-pcl 
   #+lispworks #:hcl 
   #+allegro #:mop 
   #+clisp #:clos
   #:class-slots #:slot-definition-name)
  (:documentation "Implements generic clone operation."))

(in-package :copy)

;; Source: http://stackoverflow.com/questions/11067899/is-there-a-generic-method-for-cloning-clos-objects
;; NOTE: This is a shallow copy. Add an around method for classes which need deep copy of the slots.
(defgeneric copy-instance (object &rest initargs &key &allow-other-keys)
  (:method ((object atom) &rest initargs &key &allow-other-keys) 
    (declare (ignore initargs))
    object)
  (:method ((object cons) &rest initargs &key &allow-other-keys) 
    (declare (ignore initargs))
    (copy-seq object))
  (:documentation "Makes and returns a shallow copy of OBJECT.

  An uninitialized object of the same class as OBJECT is allocated by
  calling ALLOCATE-INSTANCE.  For all slots returned by
  CLASS-SLOTS, the returned object has the
  same slot values and slot-unbound status as OBJECT.

  REINITIALIZE-INSTANCE is called to update the copy with INITARGS.")
  (:method ((object standard-object) &rest initargs &key &allow-other-keys)
    (let* ((class (class-of object))
           (copy (allocate-instance class)))
      (dolist (slot-name (mapcar #'slot-definition-name (class-slots class)))
	(when (slot-boundp object slot-name)
	  (setf (slot-value copy slot-name)
		(slot-value object slot-name))))
      (apply #'reinitialize-instance copy initargs))))

#|
;; A stripped-down version of shallow copy
;; Source: http://stackoverflow.com/questions/11067899/is-there-a-generic-method-for-cloning-clos-objects
(defun shallow-copy-object (original)
  (let* ((class (class-of original))
         (copy (allocate-instance class)))
    (dolist (slot (mapcar #'slot-definition-name (class-slots class)))
      (when (slot-boundp original slot)
        (setf (slot-value copy slot)
              (slot-value original slot))))
    copy))
|#
