;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "TERM"
  (:use :cl :monom :ring)
  (:export "TERM" 
	   "MAKE-TERM-VARIABLE"
	   )
  (:documentation "This package implements class TERM. A term is a
product of a scalar and powers of some variables, such as
5*X^2*Y^3. The part of the term without the coefficient is a monomial
X^2*Y^3, which is represented by class MONOM, provided by the :MONOM
package. In this implementation, a TERM specializes MONOL. Also, a
monomial can be considered a TERM whose coefficient is the unit
element (1) of the underlying ring. The generic method CHANGE-CLASS
can be used to convert between a MONOM and a TERM, observing this
convention."))

(in-package :term)

(proclaim '(optimize (speed 3) (space 0) (safety 0) (debug 0)))

(defclass term (monom)
  ((coeff :initarg :coeff :accessor term-coeff
	  :documentation "The coefficient."))
  (:default-initargs :dimension nil :exponents nil :coeff nil)
  (:documentation "Implements a term, i.e. a product of a ring element
and powers of some variables, such as 5*X^2*Y^3."))

(defmethod r-coeff ((self term))
  (term-coeff self))

(defmethod (setf r-coeff) (new-value (self term))
  (setf (term-coeff self) new-value))

(defmethod r-equalp ((term1 term) (term2 term))
  (and (r-equalp (term-coeff term1) (term-coeff term2))
       (call-next-method)))

(defmethod print-object ((self term) stream)
  (format stream "#<TERM DIMENSION=~A EXPONENTS=~A COEFF=~A>"
	  (r-dimension self)
	  (r-exponents self)
	  (r-coeff self)))

(defmethod shared-initialize ((self term) slot-names
			      &rest 
				initargs 
			      &key
				coeff
				&allow-other-keys)
  (declare (ignore initargs))
  (if (eq slot-names t) (setf slot-names '(coeff)))
  (dolist (slot-name slot-names)
    (case slot-name
      (coeff
       (setf (slot-value self 'coeff) coeff)))))

(defmethod update-instance-for-different-class :after ((old monom) (new term) &key)
  ;; Changing an instance of class MONOM to class TERM may also
  ;; happen when OLD is an instance of TERM, in which case the
  ;; value of the coefficient should be preserved.
  (unless (slot-boundp new 'coeff)
    (setf (term-coeff new) 1))
  new)

#|
(defun make-term-variable (nvars pos
			   &optional
			     (power 1)
			     (coeff 1))
  "Construct a term in the polynomial ring RING[X[0],X[1],X[2],...X[NVARS-1]]
over the ring RING which represents a single variable. It assumes
number of variables NVARS and the variable is at position
POS. Optionally, the variable may appear raised to power POWER.
Optionally, the term may appear with an arbitrary coefficient, which
defaults to the unit of the RING."
  (declare (type fixnum nvars pos))
  (make-term :monom (make-monom-variable nvars pos power)
	     :coeff coeff))

|#

(defmethod multiply-by ((self term) (other term))
  "Destructively multiply terms SELF and OTHER and store the result into SELF.
It returns SELF."
  (setf (r-coeff self) (multiply-by (r-coeff self) (r-coeff other))))

(defmethod divide-by ((self term) (other term))
  "Destructively divide term SELF by OTHER and store the result into SELF.
It returns SELF."
  (setf (r-coeff self) (divide-by (r-coeff self) (r-coeff other)))
  (call-next-method))

(defmethod unary-minus ((self term))
  (setf (term-coeff self) (unary-minus (term-coeff self)))
  self)

(defmethod r* ((term1 term) (term2 term))
  "Non-destructively multiply TERM1 by TERM2."
  (multiply-by (copy-instance TERM1) (copy-instance TERM2)))

(defmethod r-zerop ((self term))
  (r-zerop (term-coeff self)))

#|

(defun term->cons (term) 
  "A human-readable representation of a term as a cons (MONOM . COEFF)."
  (declare (type term term))
  (cons (monom->list (term-monom term)) (term-coeff term)))

|#
