;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;----------------------------------------------------------------
;; This package implements BASIC OPERATIONS ON MONOMIALS
;;----------------------------------------------------------------
;; DATA STRUCTURES: Conceptually, monomials can be represented as lists:
;;
;; 	monom:	(n1 n2 ... nk) where ni are non-negative integers
;;
;; However, lists may be implemented as other sequence types,
;; so the flexibility to change the representation should be
;; maintained in the code to use general operations on sequences
;; whenever possible. The optimization for the actual representation
;; should be left to declarations and the compiler.
;;----------------------------------------------------------------
;; EXAMPLES: Suppose that variables are x and y. Then
;;
;; 	Monom x*y^2 ---> (1 2)
;;
;;----------------------------------------------------------------

(defpackage "MONOM"
  (:use :cl)
  (:export "MONOM"
	   "EXPONENT"
	   "MAKE-MONOM" 
	   "MAKE-MONOM-VARIABLE" 
	   "MONOM-ELT" 
	   "MONOM-DIMENSION" 
	   "MONOM-TOTAL-DEGREE"
	   "MONOM-SUGAR" 
	   "MONOM-DIV"
	   "MONOM-MUL" 
	   "MONOM-DIVIDES-P"
	   "MONOM-DIVIDES-MONOM-LCM-P"
	   "MONOM-LCM-DIVIDES-MONOM-LCM-P"
	   "MONOM-LCM-EQUAL-MONOM-LCM-P"
	   "MONOM-DIVISIBLE-BY-P"
	   "MONOM-REL-PRIME-P"
	   "MONOM-EQUAL-P"
	   "MONOM-LCM"
	   "MONOM-GCD"
	   "MONOM-DEPENDS-P"
	   "MONOM-MAP"
	   "MONOM-APPEND"
	   "MONOM-CONTRACT"
	   "MONOM->LIST"))

(in-package :monom)

(deftype exponent ()
  "Type of exponent in a monomial."
  'fixnum)

(deftype monom (&optional dim)
  "Type of monomial."
  `(simple-array exponent (,dim)))

;; If a monomial is redefined as structure with slot EXPONENTS, the function
;; below can be the BOA constructor.
(defun make-monom (&key 
		     (dimension nil dimension-suppied-p)
		     (initial-exponents nil initial-exponents-supplied-p)
		     (initial-exponent  nil initial-exponent-supplied-p)
		   &aux
		     (dim (cond (dimension-suppied-p dimension)
				(initial-exponents-supplied-p (length initial-exponents))
				(t (error "You must provide DIMENSION nor INITIAL-EXPONENTS"))))
		     (monom (cond 
			      ;; when exponents are supplied
			      (initial-exponents-supplied-p
			       (make-array (list dim) :initial-contents initial-exponents
					   :element-type 'exponent))
			      ;; when all exponents are to be identical
			      (initial-exponent-supplied-p
			       (make-array (list dim) :initial-element initial-exponent
					   :element-type 'exponent))
			      ;; otherwise, all exponents are zero
			      (t 
			       (make-array (list dim) :element-type 'exponent :initial-element 0)))))
  "A constructor (factory) of monomials. If DIMENSION is given, a sequence of
DIMENSION elements of type EXPONENT is constructed, where individual
elements are the value of INITIAL-EXPONENT, which defaults to 0.
Alternatively, all elements may be specified as a list
INITIAL-EXPONENTS."
 monom)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Operations on monomials
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun monom-dimension (m)
  (length m))

(defmacro monom-elt (m index)
  "Return the power in the monomial M of variable number INDEX."
  `(elt ,m ,index))

(defun monom-total-degree (m &optional (start 0) (end (monom-dimension m)))
  "Return the todal degree of a monomoal M. Optinally, a range
of variables may be specified with arguments START and END."
  (reduce #'+ m :start start :end end))

(defun monom-sugar (m &aux (start 0) (end (monom-dimension m)))
  "Return the sugar of a monomial M. Optinally, a range
of variables may be specified with arguments START and END."
  (monom-total-degree m start end))

(defun monom-div (m1 m2 &aux (result (copy-seq m1)))
  "Divide monomial M1 by monomial M2."
  (map-into result #'- m1 m2))

(defun monom-mul (m1 m2  &aux (result (copy-seq m1)))
  "Multiply monomial M1 by monomial M2."
  (map-into result #'+ m1 m2))

(defun monom-divides-p (m1 m2)
  "Returns T if monomial M1 divides monomial M2, NIL otherwise."
  (every #'<= m1 m2))

(defun monom-divides-monom-lcm-p (m1 m2 m3)
  "Returns T if monomial M1 divides MONOM-LCM(M2,M3), NIL otherwise."
  (every #'(lambda (x y z) (<= x (max y z))) 
	 m1 m2 m3))

(defun monom-lcm-divides-monom-lcm-p (m1 m2 m3 m4)
  "Returns T if monomial MONOM-LCM(M1,M2) divides MONOM-LCM(M3,M4), NIL otherwise."
  (every #'(lambda (x y z w) (<= (max x y) (max z w))) 
	 m1 m2 m3 m4))
	 

(defun monom-lcm-equal-monom-lcm-p (m1 m2 m3 m4)
  "Returns T if monomial MONOM-LCM(M1,M2) equals MONOM-LCM(M3,M4), NIL otherwise."
  (every #'(lambda (x y z w) (= (max x y) (max z w)))
	 m1 m2 m3 m4))


(defun monom-divisible-by-p (m1 m2)
  "Returns T if monomial M1 is divisible by monomial M2, NIL otherwise."
  (every #'>= m1 m2))

(defun monom-rel-prime-p (m1 m2)
  "Returns T if two monomials M1 and M2 are relatively prime (disjoint)."
  (every #'(lambda (x y) (zerop (min x y))) m1 m2))

(defun monom-equal-p (m1 m2)
  "Returns T if two monomials M1 and M2 are equal."
  (every #'= m1 m2))

(defun monom-lcm (m1 m2 &aux (result (copy-seq m1)))
  "Returns least common multiple of monomials M1 and M2."
  (map-into result #'max m1 m2))

(defun monom-gcd (m1 m2 &aux (result (copy-seq m1)))
  "Returns greatest common divisor of monomials M1 and M2."
  (map-into result #'min m1 m2))

(defun monom-depends-p (m k)
  "Return T if the monomial M depends on variable number K."
  (plusp (monom-elt m k)))

(defmacro monom-map (fun m &rest ml &aux (result `(copy-seq ,m)))
  `(map-into ,result ,fun ,m ,@ml))

(defmacro monom-append (m1 m2)
  `(concatenate 'monom ,m1 ,m2))

(defmacro monom-contract (m k)
  "Remove the first K variables from monomial M. Equivalent to setting
these variables to 1."
  `(setf ,m (subseq ,m ,k)))

(defun make-monom-variable (nvars pos &optional (power 1)
			    &aux (m (make-monom :dimension nvars)))
  "Construct a monomial in the polynomial ring
RING[X[0],X[1],X[2],...X[NVARS-1]] over the (unspecified) ring RING
which represents a single variable. It assumes number of variables
NVARS and the variable is at position POS. Optionally, the variable
may appear raised to power POWER. "
  (setf (monom-elt m pos) power)
  m)

(defun monom->list (m)
  "A human-readable representation of a monomial M as a list of exponents."  
  (coerce m 'list))
