;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Standard postprocessing of Grobner bases:
;; - reduction
;; - minimization
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "GB-POSTPROCESSING"
  (:use :cl :monomial :division :polynomial :ring :ring-and-order)
  (:export "REDUCTION" "MINIMIZATION"))

(in-package :gb-postprocessing)

(defun reduction (ring-and-order plist
		  &aux
		    (ring (ro-ring ring-and-order)))
  "Reduce a list of polynomials PLIST, so that non of the terms in any of
the polynomials is divisible by a leading monomial of another
polynomial.  Return the reduced list."
  (declare (type ring-and-order ring-and-order))
  (do ((q plist)
       (found t))
      ((not found)
       (mapcar #'(lambda (x) (poly-primitive-part ring x)) q))
    ;;Find p in Q such that p is reducible mod Q\{p}
    (setf found nil)
    (dolist (x q)
      (let ((q1 (remove x q)))
	(multiple-value-bind (h c div-count)
	    (normal-form ring-and-order x q1 nil #| not a top reduction! |# )
	  (declare (ignore c))
	  (unless (zerop div-count)
	    (setf found t q q1)
	    (unless (poly-zerop h)
	      (setf q (nconc q1 (list h))))
	    (return)))))))

(defun minimization (p)
  "Returns a sublist of the polynomial list P spanning the same
monomial ideal as P but minimal, i.e. no leading monomial
of a polynomial in the sublist divides the leading monomial
of another polynomial."
  (do ((q p)
       (found t))
      ((not found) q)
    (declare (type poly q))
    ;;1) Find p in Q such that lm(p) is in LM(Q\{p})
    ;;2) Set Q <- Q\{p}
    (setf found nil)
    ;; NOTE: Below we rely on the fact that NIL is not of type POLY
    (let ((x (find-if 
	      #'(lambda (y) 
		  (find-if #'(lambda (p) 
			       (monom-divides-p 
				(poly-lm p) 
				(poly-lm y)))
			   (remove y q)))
	      q)))
      (when x
	(setf found t
	      q (delete x q))))))

