;;; -*-  Mode: Lisp; Package: Maxima; Syntax: Common-Lisp; Base: 10 -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

(load "ngrobner.asd")
(asdf:load-system :ngrobner)

(defpackage #:ngrobner-tests
  (:use :cl :it.bese.fiveam 
	:ngrobner :priority-queue :monomial
	:utils :order :ring :term :ring-and-order 
	:termlist :polynomial
	:priority-queue 
	:division
	)
  )

(in-package #:ngrobner-tests)

(def-suite ngrobner-suite 
    :description "New Groebner Package Suite")

(in-suite ngrobner-suite)

#+nil
(test dummy-test
  "Makelist"
  (is (= (+ 2 2)) "2 plus 2 wasn't equal to 4 (using #'= to test equality)")
  (is (= 0 (+ -1 1)))
  (signals
      (error "Trying to add 4 to FOO didn't signal an error")
    (+ 'foo 4))
  (is (= 0 (+ 1 1)) "this should have failed"))

(test makelist-1 
  "makelist-1 test"
  (is (equal (makelist-1 (* 2 i) i 0 10) '(0 2 4 6 8 10 12 14 16 18 20)))
  (is (equal (makelist-1 (* 2 i) i 0 10 3) '(0 6 12 18))))

(test makelist
  "makelist"
  (is (equal (makelist (+ (* i i) (* j j)) (i 1 4) (j 1 i)) '(2 5 8 10 13 18 17 20 25 32)))
  (is (equal (makelist (list i j '---> (+ (* i i) (* j j))) (i 1 4) (j 1 i))
	     '((1 1 ---> 2) (2 1 ---> 5) (2 2 ---> 8) (3 1 ---> 10) (3 2 ---> 13)
	       (3 3 ---> 18) (4 1 ---> 17) (4 2 ---> 20) (4 3 ---> 25) (4 4 ---> 32)))))

(test monom
  "monom"
  (is (every #'= (make-monom :dimension 3) '(0 0 0)) "Trivial monomial is a vector of 0's")
  (is (every #'= (make-monom :initial-exponents '(1 2 3)) '(1 2 3)) "Monomial with powers 1,2,3")
  (let ((p (make-monom :initial-exponents '(1 2 3))))
    (is (every #'= (monom-map (lambda (x) x) p) '(1 2 3)))))

  
(test order
  "order"
  (let ((p (make-monom :initial-exponents '(1 3 2)))
	(q (make-monom :initial-exponents '(1 2 3))))
    (is-true (lex>  p q)) 
    (is-true (grlex>  p q)) 
    (is-true (revlex>  p q)) 
    (is-true (grevlex>  p q)) 
    (is-false (invlex>  p q))))

(test elim-order
  "elimination order"
  (let* ((p (make-monom :initial-exponents '(1 2 3)))
	 (q (make-monom :initial-exponents '(4 5 6)))
	 (elim-order-factory (make-elimination-order-factory))
	 (elim-order-1 (funcall elim-order-factory 1))
	 (elim-order-2 (funcall elim-order-factory 2)))
    (is-false (funcall elim-order-1 p q))
    (is-false (funcall elim-order-2 p q))))

(test term
  "term"
  (let* ((m1 (make-monom :initial-exponents '(1 2 3)))
	 (m2 (make-monom :initial-exponents '(3 5 2)))
	 (m3 (monom-mul m1 m2))
	 (t1 (make-term m1 7))
	 (t2 (make-term m2 9))
	 (t3 (make-term m3 (* 7 9))))
    (is (equalp (term-mul *ring-of-integers* t1 t2) t3))))

(test termlist
  "termlist"
  (let* ((t1 (make-term  (make-monom :initial-exponents '(1 2 3)) 7))
	 (t2 (make-term  (make-monom :initial-exponents '(3 5 2)) 9))
	 (t11 (make-term (make-monom :initial-exponents '(2 4 6)) 49))
	 (t12 (make-term (make-monom :initial-exponents '(4 7 5)) 126))
	 (t22 (make-term (make-monom :initial-exponents '(6 10 4)) 81))
	 (p (list t2 t1))
	 (p-sq (list t22 t12 t11))
	 (ring-and-order (make-ring-and-order))
	 (q (termlist-expt ring-and-order p 2)))
    (is-true (equalp q p-sq))))

(test poly
  "poly"
  (let* ((t1 (make-term  (make-monom :initial-exponents '(1 2 3)) 7))
	 (t2 (make-term  (make-monom :initial-exponents '(3 5 2)) 9))
	 (t11 (make-term (make-monom :initial-exponents '(2 4 6)) 49))
	 (t12 (make-term (make-monom :initial-exponents '(4 7 5)) 126))
	 (t22 (make-term (make-monom :initial-exponents '(6 10 4)) 81))
	 (p (make-poly-from-termlist (list t2 t1)))
	 (p-sq (make-poly-from-termlist (list t22 t12 t11)))
	 (ring-and-order (make-ring-and-order))
	 (q (poly-expt ring-and-order p 2)))
    (is-true (equalp q p-sq))))


(test coerce-to-infix
  "Conversion to infix form"
  (is (equal 
       (coerce-to-infix :term (make-term-variable *ring-of-integers* 5 3) '(x y z w u v))
       '(* 1 (EXPT X 0) (EXPT Y 0) (EXPT Z 0) (EXPT W 1) (EXPT U 0)))))

(test priority-queue
  "Priority queue"
  (let ((q (make-priority-queue)))
    (priority-queue-insert q 7)
    (priority-queue-insert q 8)
    (is (= (priority-queue-size q) 3) "Note that there is always a dummy element in the queue.")
    (is (equalp (priority-queue-heap q) #(0 7 8)))
    (is (= (priority-queue-remove q) 7))
    (is (= (priority-queue-remove q) 8))
    (is-true (priority-queue-empty-p q))
    (signals
	(error "Empty queue.")
      (priority-queue-remove q))))

;;
;; Currently parser cannot be tested, as it relies on many maxima functions
;; to parse a polynomial expression.
;;
#|
(test parser
  "Parser"
  (let (($f '((MLIST SIMP) ((MPLUS SIMP) $X ((MTIMES SIMP) -1 $Y)) ((MPLUS SIMP) $X $Y)))
	($v '((MLIST SIMP) $X $Y)))
    (is-true (parse-poly-list $f $v))))
|#

(test infix-print 
  "Infix printer"
  (is (string= (infix-print '(+ x y) nil) "X+Y"))
  (is (string= (infix-print '(expt x 3) nil) "X^3"))
  (is (string= (infix-print '(+ 1 (expt x 3)) nil) "1+(X^3)"))
  (is (string= (infix-print '(* x y) nil) "X*Y"))
  (is (string= (infix-print '(* x (expt y 2)) nil) "X*(Y^2)")))

(test infix
  "Infix parser"
  (is (equal '#I( x^2 + y^2 ) '(+ (expt x 2) (expt y 2))))
  (is (equal '#I( [ x, y ] ) '(:[ X Y)))
  (is (equal '#I( x + y) '(+ x y)))
  (is (equal '#I( x^3 ) '(expt x 3)))
  (is (equal '#I( 1 + x^3) '(+ 1 (expt x 3))))
  (is (equal '#I( x * y^2 ) '(* x (expt y 2)))))

(test poly-reader
  "Polynomial reader"
  (is (equalp (with-input-from-string (s "X^2-Y^2+(-4/3)*U^2*W^3-5")
		(read-infix-form :stream s))
	      '(+ (- (EXPT X 2) (EXPT Y 2)) (* (- (/ 4 3)) (EXPT U 2) (EXPT W 3)) (- 5))))
  (is (equalp (string->alist "X^2-Y^2+(-4/3)*U^2*W^3-5" '(x y u w))
	      '(((2 0 0 0) . 1)
		((0 2 0 0) . -1) 
		((0 0 2 3) . -4/3)
		((0 0 0 0) . -5))))
  (is (equalp (string->alist "[x^2-y^2+(-4/3)*u^2*w^3-5,y]" '(x y u w))
	      '(:[ 
		(((2 0 0 0) . 1) ((0 2 0 0) . -1) ((0 0 2 3) . -4/3) ((0 0 0 0) . -5))
		(((0 1 0 0) . 1)))))
  (let ((p (make-poly-from-termlist (list (make-term (make-monom :initial-exponents '(2 0)) 1)
					  (make-term (make-monom :initial-exponents '(0 2)) 2)))))
    (is (equalp (with-input-from-string (s "x^2+2*y^2") 
		  (read-poly '(x y) :stream s))
		p))
    (is (equalp (string->poly "x^2+2*y^2" '(x y)) p))))

(test division
  "Division in polynomial ring"
  (let* ((f (string->poly "x^2" '(x y)))
	 (y-sq (string->poly "y^2" '(x y)))
	 (fl (cdr (string->poly "[x+y,x-2*y]" '(x y))))
	 (ring *ring-of-integers*)
	 (order #'lex>)
	 (ring-and-order (make-ring-and-order :ring ring :order order)))
    (is (equalp (normal-form ring-and-order f fl) y-sq))
    (is-true (print (multiple-value-list (poly-pseudo-divide ring-and-order f fl))))
    (is-false (buchberger-criterion ring-and-order fl))))


(run! 'ngrobner-suite)
(format t "All tests done!~%")


