;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

(defpackage #:5am-buchberger
  (:use :cl :it.bese.fiveam :monom :polynomial :infix :symbolic-polynomial :division :priority-queue :buchberger :grobner-debug))

(in-package :5am-buchberger)

(def-suite buchberger-suite 
    :description "Buchberger algorithm suite")

(in-suite buchberger-suite)

(def-fixture buchberger-context ()
  (let ((fl (cdr (string->poly "[x+y,x-2*y]" '(x y))))
	(gb (cdr (string->poly "[x+y,x-2*y,y]" '(x y)))))
    (&body)))


(test buchberger
  "Buchberger algorithm"
  (with-fixture buchberger-context ()
    (is-true (grobner-test gb fl))
    (is (every #'universal-equalp (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; poly_grobner([x-3*u-3*u*v^2+u^3,y-3*v-3*u^2*v+v^3,z-3*u^2+3*v^2],[u,v,x,y,z]);
(def-fixture buchberger-enneper-surface-context ()
  (let ((fl (cdr (string->poly "[x-3*u-3*u*v^2+u^3,y-3*v-3*u^2*v+v^3,z-3*u^2+3*v^2]" '(u v x y z))))
	(gb (cdr (string->poly "[x-3*u*v^2+u^3-3*u,y+v^3-3*u^2*v-3*v,z+3*v^2-3*u^2,
       (-u*z)-3*x+6*u*v^2+9*u,(-v*z)+y-2*v^3-3*v,z^2+6*v^2*z-9*z-9*v*y+9*u*x,
       4*u*v*z-3*u*y+3*v*x,2*u*z^2+6*x*z-18*u*z-9*u*v*y+9*v^2*x,
       (-8*u*z^3)-24*x*z^2+72*u*z^2-36*v^2*x*z+27*u*y^2-27*v*x*y,
       z^3+18*v^2*z^2-18*z^2-54*v*y*z+54*v^2*z+81*z+27*y^2-27*x^2,
       (-4*z^4)+48*z^3-108*v*y*z^2+108*z^2+135*y^2*z+324*v*y*z+108*x^2*z
               -1296*v^2*z-1944*z-243*v^2*y^2-648*y^2+243*v^2*x^2+648*x^2,
       8*v*z^3-9*y*z^2+72*v*z^2+54*v^2*y*z-27*y*z-27*v*y^2+27*v*x^2,
       (-8*v*z^4)+12*y*z^3-96*v*z^3-216*v*z^2-135*v*y^2*z+324*y*z-27*v*x^2*z
                 +81*y^3+81*v*y^2-81*x^2*y-81*v*x^2,
       (-64*v*z^6)+120*y*z^5-1152*v*z^5+288*y*z^4-5184*v*z^4-648*v*y^2*z^3
                  -216*y*z^3+6912*v*z^3+81*y^3*z^2-9720*v*y^2*z^2
                  -1539*x^2*y*z^2+31104*y*z^2+62208*v*z^2+8505*y^3*z
                  +46656*v*y^2*z-8505*x^2*y*z-93312*y*z+729*v*y^4-23328*y^3
                  -1458*v*x^2*y^2-23328*v*y^2+23328*x^2*y+729*v*x^4
                  +23328*v*x^2,
       8*z^6-72*z^5+648*v*y*z^4-648*z^4-945*y^2*z^3+5184*v*y*z^3-189*x^2*z^3
            +5832*z^3+972*y^2*z^2+17496*v*y*z^2-2430*x^2*z^2+8748*v*y^3*z
            -19683*y^2*z+2187*x^2*z-5103*y^4-4374*v*y^3+5832*x^2*y^2
            +4374*v*x^2*y-729*x^4,
       8*z^7-48*z^6+648*v*y*z^5-864*z^5-945*y^2*z^4+5832*v*y*z^4-189*x^2*z^4
            +3888*z^4+81*y^2*z^3+17496*v*y*z^3-2997*x^2*z^3+17496*z^3
            +8748*v*y^3*z^2-16767*y^2*z^2+17496*v*y*z^2-5103*x^2*z^2
            -5103*y^4*z+5832*x^2*y^2*z-6561*y^2*z-729*x^4*z+6561*x^2*z
            -2187*y^4+4374*x^2*y^2-2187*x^4,
       64*z^9-10368*z^7+1296*y^2*z^6-1296*x^2*z^6-34992*y^2*z^5-34992*x^2*z^5
             +419904*z^5+174960*y^2*z^4-174960*x^2*z^4-10935*y^4*z^3
             -56862*x^2*y^2*z^3+314928*y^2*z^3-10935*x^4*z^3+314928*x^2*z^3
             +118098*y^4*z^2-118098*x^4*z^2+59049*y^4*z-118098*x^2*y^2*z
             +59049*x^4*z+19683*y^6-59049*x^2*y^4+59049*x^4*y^2-19683*x^6]" '(u v x y z)))))
    (&body)))
    

(test buchberger-enneper-surface
  "Buchberger algorithm - Enneper surface"
  (with-fixture buchberger-enneper-surface-context ()
    (is-true (grobner-test gb fl))
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; Cyclic roots of degree 5
;; poly_reduced_grobner([x+y+z+u+v,x*y+y*z+z*u+u*v+v*x,x*y*z+y*z*u+z*u*v+u*v*x+v*x*y,x*y*z*u+y*z*u*v+z*u*v*x+u*v*x*y+v*x*y*z,x*y*z*u*v-1],[u,v,x,y,z]);
(def-fixture buchberger-cyclic-5-context ()
  (let ((fl 
	 (cdr (string->poly 
	       "[x+y+z+u+v,x*y+y*z+z*u+u*v+v*x,x*y*z+y*z*u+z*u*v+u*v*x+v*x*y,x*y*z*u+y*z*u*v+z*u*v*x+u*v*x*y+v*x*y*z,x*y*z*u*v-1]"
	       '(u v x y z))))
	(gb (cdr (string->poly "[z+y+x+v+u,
       8*z^12+2*y*z^11+979*z^7+231*y*z^6-55*y^2*z^5-987*z^2-233*y*z+55*y^2,
       z^15+122*z^10-122*z^5-1,
       1467*z^12+566*y*z^11+178981*z^7+69003*y*z^6+550*y^3*z^4-275*y^4*z^3
                -1650*y^5*z^2-179073*z^2-550*y^6*z-69019*y*z-825*v*z-275*y^2
                -275*v^2,(-z^11)-143*z^6-55*v*z^5+144*z+55*v,
       (-346*z^12)-124*y*z^11-42218*z^7-15092*y*z^6-275*y^3*z^4+275*y^4*z^3
                  +440*y^5*z^2+42124*z^2+110*y^6*z+15106*y*z+275*v*z-275*v*y,
       867*z^12+334*y*z^11+105776*z^7+40722*y*z^6+275*y^3*z^4-275*y^4*z^3
               -1045*y^5*z^2-105873*z^2-330*y^6*z-40726*y*z+550*x*z-275*v*z
               -550*y^2+275*x^2+275*v*x,
       (-568*z^13)-232*y*z^12-69289*z^8-28336*y*z^7-550*y^4*z^4+550*y^5*z^3
                  +69307*z^3+275*y^6*z^2+28018*y*z^2-550*x*z^2+550*y^2*z
                  +550*x^2*z+275*x^3,(-z^11)-143*z^6-55*x*z^5+144*z+55*x,
       1121*z^12+442*y*z^11+136763*z^7+53911*y*z^6+275*y^3*z^4-1210*y^5*z^2
                -136674*z^2-440*y^6*z-53913*y*z+275*x*z-275*y^2-275*x*y,
       1042*z^12+398*y*z^11+127116*z^7+48554*y*z^6-55*y^5*z^2-128103*z^2
                -165*y^6*z-48787*y*z-55*y^7+55*y^2]" '(u v x y z)))))
    (&body)))


(test buchberger-cyclic-5
  "Buchberger algorithm - cyclic 5"
  (with-fixture buchberger-cyclic-5-context ()
    (is-true (grobner-test gb fl))
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )

;; A tough example learned from Cox
;; poly_grobner([x^5+y^4+z^3-1,x^3+y^3+z^2-1], [x,y,z]);
(def-fixture buchberger-cox-tough-context ()
  (let ((fl (cdr (string->poly "[x^5+y^4+z^3-1,x^3+y^3+z^2-1]" '(x y z))))
	(gb (cdr (string->poly "[z^2+y^3+x^3-1,z^3-x^2*z^2+y^4-x^2*y^3+x^2-1,
       (-z^4)-x*z^3-2*y^3*z^2+2*z^2-y^6-x*y^4+2*y^3+x-1,
       (-z^10)+z^9-5*y^3*z^8+5*z^8-10*y^6*z^6+3*y^4*z^6+20*y^3*z^6-13*z^6
              -10*y^9*z^4+30*y^6*z^4-30*y^3*z^4+10*z^4+3*y^8*z^3-6*y^4*z^3
              +3*z^3-5*y^12*z^2+20*y^9*z^2-30*y^6*z^2+20*y^3*z^2-5*z^2-y^15
              +6*y^12-10*y^9-3*y^8+10*y^6+3*y^4-5*y^3,
       (-44425*z^21)-44425*x*z^20-193795*z^20-88850*y^3*z^19+266550*y*z^19
                    -327070*x*z^19-1274354*z^19+44425*y^4*z^18-32190*y^3*z^18
                    +444250*y^2*z^18+451970*y*z^18-1278214*x*z^18-3527528*z^18
                    -44425*y^6*z^17+1526545*y^4*z^17-1842423*y^3*z^17
                    +1227150*y^2*z^17-1140996*y*z^17-2698855*x*z^17
                    -1109995*z^17+88850*y^7*z^16+783555*y^6*z^16
                    +1954700*y^5*z^16+4467129*y^4*z^16-9605029*y^3*z^16
                    -752630*y^2*z^16-2972806*y*z^16-2572042*x*z^16
                    +13565093*z^16-44425*y^8*z^15+2697690*y^7*z^15
                    +456911*y^6*z^15+7105380*y^5*z^15+2647468*y^4*z^15
                    -10602997*y^3*z^15-7006014*y^2*z^15+1218640*y*z^15
                    +2619449*x*z^15+22302653*z^15+44425*y^10*z^14
                    +888500*y^9*z^14+2915955*y^8*z^14+9160898*y^7*z^14
                    -7506655*y^6*z^14+5800536*y^5*z^14-3245086*y^4*z^14
                    +20587543*y^3*z^14-8783974*y^2*z^14+3837574*y*z^14
                    +10650408*x*z^14-7954033*z^14-88850*y^11*z^13
                    +1881945*y^10*z^13+1921200*y^9*z^13+11847046*y^8*z^13
                    +9782204*y^7*z^13-12668822*y^6*z^13-10997536*y^5*z^13
                    +1090138*y^4*z^13+54430174*y^3*z^13-1436450*y^2*z^13
                    -11018460*y*z^13+11493837*x*z^13-56030677*z^13
                    +266550*y^12*z^12+1744810*y^11*z^12+6728289*y^10*z^12
                    -187950*y^9*z^12+15235952*y^8*z^12-505866*y^7*z^12
                    +10778914*y^6*z^12-25975004*y^5*z^12+899768*y^4*z^12
                    +13274582*y^3*z^12+4950830*y^2*z^12-21334144*y*z^12
                    -2616810*x*z^12-42908546*z^12-44425*y^14*z^11
                    +444250*y^13*z^11+718520*y^12*z^11+7686377*y^11*z^11
                    +9990070*y^10*z^11-1596036*y^9*z^11-1342215*y^8*z^11
                    -8841150*y^7*z^11+40955979*y^6*z^11-24957052*y^5*z^11
                    -36144372*y^4*z^11-82735585*y^3*z^11+13378064*y^2*z^11
                    +12799086*y*z^11-16744470*x*z^11+42041860*z^11
                    +339305*y^14*z^10+1671400*y^13*z^10+910274*y^12*z^10
                    +11626231*y^11*z^10+7801347*y^10*z^10+7831314*y^9*z^10
                    -26096155*y^8*z^10-16474014*y^7*z^10+11243399*y^6*z^10
                    -17808216*y^5*z^10-61462426*y^4*z^10-89174777*y^3*z^10
                    +30957924*y^2*z^10+55234262*y*z^10-12049440*x*z^10
                    +91874913*z^10+1628711*y^14*z^9+3140020*y^13*z^9
                    +1263518*y^12*z^9+4404103*y^11*z^9+2153415*y^10*z^9
                    +19135710*y^9*z^9-40336258*y^8*z^9-39769140*y^7*z^9
                    -68036522*y^6*z^9+12529572*y^5*z^9+7525711*y^4*z^9
                    +37559804*y^3*z^9+21952328*y^2*z^9+29962348*y*z^9
                    +4801407*x*z^9+31327296*z^9+3000105*y^14*z^8
                    +4046756*y^13*z^8+3116408*y^12*z^8-11359143*y^11*z^8
                    -9140507*y^10*z^8+5385232*y^9*z^8-38148698*y^8*z^8
                    -41664378*y^7*z^8-80668677*y^6*z^8+73310048*y^5*z^8
                    +103064435*y^4*z^8+127272309*y^3*z^8-30170622*y^2*z^8
                    -39724286*y*z^8+8854766*x*z^8-57432520*z^8
                    +2743338*y^14*z^7+3960482*y^13*z^7+5071808*y^12*z^7
                    -25510554*y^11*z^7-29133398*y^10*z^7-31118752*y^9*z^7
                    -1947773*y^8*z^7+21330064*y^7*z^7+18392944*y^6*z^7
                    +73580532*y^5*z^7+82980083*y^4*z^7+49867042*y^3*z^7
                    -52627432*y^2*z^7-56881602*y*z^7+1892306*x*z^7
                    -63266461*z^7+437586*y^14*z^6+1782242*y^13*z^6
                    +3440132*y^12*z^6-28425462*y^11*z^6-36482669*y^10*z^6
                    -46554784*y^9*z^6+58365386*y^8*z^6+86946000*y^7*z^6
                    +105699982*y^6*z^6-21170564*y^5*z^6-28598633*y^4*z^6
                    -74430398*y^3*z^6-9384858*y^2*z^6-8218528*y*z^6
                    -1980847*x*z^6-5375344*z^6-3228714*y^14*z^5
                    -2836264*y^13*z^5-2194496*y^12*z^5-9426528*y^11*z^5
                    -11662110*y^10*z^5-15507324*y^9*z^5+63744949*y^8*z^5
                    +61819454*y^7*z^5+68420946*y^6*z^5-74140892*y^5*z^5
                    -75995106*y^4*z^5-79475920*y^3*z^5+26513908*y^2*z^5
                    +26664768*y*z^5+17496*x*z^5+27268337*z^5-6067476*y^14*z^4
                    -6324094*y^13*z^4-6620078*y^12*z^4+21880364*y^11*z^4
                    +22936006*y^10*z^4+24599634*y^9*z^4+662692*y^8*z^4
                    -20296984*y^7*z^4-23355948*y^6*z^4-25675184*y^5*z^4
                    -25336908*y^4*z^4-1372204*y^3*z^4+15113990*y^2*z^4
                    +15232088*y*z^4-34992*x*z^4+14641598*z^4-5393667*y^14*z^3
                    -5404156*y^13*z^3-5499938*y^12*z^3+29810057*y^11*z^3
                    +29956543*y^10*z^3+30284550*y^9*z^3-35221464*y^8*z^3
                    -53124800*y^7*z^3-52987825*y^6*z^3+18268092*y^5*z^3
                    +18247517*y^4*z^3+36291181*y^3*z^3-2001554*y^2*z^3
                    -2010302*y*z^3+17496*x*z^3-1214234*z^3-1569800*y^14*z^2
                    -1661654*y^13*z^2-1622288*y^12*z^2+10350756*y^11*z^2
                    +10775034*y^10*z^2+10578204*y^9*z^2-15349894*y^8*z^2
                    -20474824*y^7*z^2-20457328*y^6*z^2+10504900*y^5*z^2
                    +10793584*y^4*z^2+15564220*y^3*z^2-2374910*y^2*z^2
                    +69984*x*y*z^2-2436146*y*z^2-69984*x*z^2-2619854*z^2
                    +632812*y^14*z+685300*y^13*z+632812*y^12*z-3181556*y^11*z
                    -3478988*y^10*z-3111572*y^9*z+3269036*y^8*z+5569880*y^7*z
                    +4905032*y^6*z-1388096*y^5*z-1738016*y^4*z-2796644*y^3*z
                    -139968*x*y*z+139968*y*z+139968*x*z-139968*z+501226*y^14
                    +474982*y^13+501226*y^12-2497382*y^11-2348666*y^10
                    -2532374*y^9+2453642*y^8+3756116*y^7+4088540*y^6
                    -941216*y^5-766256*y^4-2689838*y^3+69984*x*y-69984*y
                    -69984*x+69984,
       z^22+x*z^21+6*z^21+2*y^3*z^20-6*y*z^20+9*x*z^20+37*z^20-y^4*z^19
           +4*y^3*z^19-10*y^2*z^19-20*y*z^19+42*x*z^19+132*z^19+y^6*z^18
           -36*y^4*z^18+45*y^3*z^18-44*y^2*z^18+2*y*z^18+117*x*z^18+191*z^18
           -2*y^7*z^17-16*y^6*z^17-44*y^5*z^17-158*y^4*z^17+286*y^3*z^17
           -40*y^2*z^17+94*y*z^17+195*x*z^17-156*z^17+y^8*z^16-64*y^7*z^16
           -38*y^6*z^16-232*y^5*z^16-265*y^4*z^16+642*y^3*z^16+148*y^2*z^16
           +106*y*z^16+123*x*z^16-927*z^16-y^10*z^15-20*y^9*z^15-64*y^8*z^15
           -308*y^7*z^15+132*y^6*z^15-444*y^5*z^15-160*y^4*z^15+202*y^3*z^15
           +460*y^2*z^15-42*y*z^15-232*x*z^15-972*z^15+2*y^11*z^14
           -44*y^10*z^14-76*y^9*z^14-373*y^8*z^14-630*y^7*z^14+541*y^6*z^14
           -176*y^5*z^14-32*y^4*z^14-1587*y^3*z^14+544*y^2*z^14+112*y*z^14
           -678*x*z^14+822*z^14-6*y^12*z^13-36*y^11*z^13-222*y^10*z^13
           -90*y^9*z^13-856*y^8*z^13-622*y^7*z^13+414*y^6*z^13+748*y^5*z^13
           -25*y^4*z^13-2686*y^3*z^13+254*y^2*z^13+784*y*z^13-660*x*z^13
           +2938*z^13+y^14*z^12-10*y^13*z^12-26*y^12*z^12-235*y^11*z^12
           -523*y^10*z^12-18*y^9*z^12-877*y^8*z^12-156*y^7*z^12-902*y^6*z^12
           +1712*y^5*z^12+770*y^4*z^12-240*y^3*z^12-318*y^2*z^12+746*y*z^12
           +45*x*z^12+2097*z^12-6*y^14*z^11-54*y^13*z^11-54*y^12*z^11
           -590*y^11*z^11-743*y^10*z^11-138*y^9*z^11+88*y^8*z^11+558*y^7*z^11
           -1874*y^6*z^11+2096*y^5*z^11+2683*y^4*z^11+4030*y^3*z^11
           -1258*y^2*z^11-1040*y*z^11+783*x*z^11-1862*z^11-48*y^14*z^10
           -144*y^13*z^10-84*y^12*z^10-750*y^11*z^10-684*y^10*z^10
           -684*y^9*z^10+1680*y^8*z^10+1686*y^7*z^10-26*y^6*z^10+1356*y^5*z^10
           +3037*y^4*z^10+4238*y^3*z^10-2018*y^2*z^10-2768*y*z^10+744*x*z^10
           -4500*z^10-136*y^14*z^9-256*y^13*z^9-150*y^12*z^9-308*y^11*z^9
           -223*y^10*z^9-1018*y^9*z^9+2968*y^8*z^9+2916*y^7*z^9+3544*y^6*z^9
           -1244*y^5*z^9-565*y^4*z^9-1108*y^3*z^9-1108*y^2*z^9-1716*y*z^9
           +115*x*z^9-2504*z^9-219*y^14*z^8-342*y^13*z^8-276*y^12*z^8
           +711*y^11*z^8+795*y^10*z^8-88*y^9*z^8+2781*y^8*z^8+2318*y^7*z^8
           +4061*y^6*z^8-4336*y^5*z^8-5025*y^4*z^8-5533*y^3*z^8+1306*y^2*z^8
           +1318*y*z^8-300*x*z^8+1617*z^8-210*y^14*z^7-336*y^13*z^7
           -366*y^12*z^7+1770*y^11*z^7+2058*y^10*z^7+1806*y^9*z^7+314*y^8*z^7
           -1112*y^7*z^7-270*y^6*z^7-4180*y^5*z^7-4922*y^4*z^7-3486*y^3*z^7
           +2552*y^2*z^7+2770*y*z^7-240*x*z^7+3228*z^7-57*y^14*z^6
           -168*y^13*z^6-254*y^12*z^6+2019*y^11*z^6+2449*y^10*z^6+2738*y^9*z^6
           -2900*y^8*z^6-4496*y^7*z^6-4787*y^6*z^6-324*y^5*z^6-497*y^4*z^6
           +1823*y^3*z^6+1334*y^2*z^6+1442*y*z^6-64*x*z^6+1614*z^6
           +198*y^14*z^5+132*y^13*z^5+66*y^12*z^5+922*y^11*z^5+1232*y^10*z^5
           +1542*y^9*z^5-3648*y^8*z^5-4116*y^7*z^5-4584*y^6*z^5+2748*y^5*z^5
           +2820*y^4*z^5+3882*y^3*z^5-418*y^2*z^5-398*y*z^5-378*z^5
           +408*y^14*z^4+384*y^13*z^4+360*y^12*z^4-792*y^11*z^4-672*y^10*z^4
           -552*y^9*z^4-1472*y^8*z^4-656*y^7*z^4-848*y^6*z^4+2320*y^5*z^4
           +2368*y^4*z^4+1768*y^3*z^4-872*y^2*z^4-872*y*z^4-872*z^4
           +422*y^14*z^3+418*y^13*z^3+414*y^12*z^3-1642*y^11*z^3-1622*y^10*z^3
           -1602*y^9*z^3+718*y^8*z^3+1916*y^7*z^3+1884*y^6*z^3+512*y^5*z^3
           +520*y^4*z^3-642*y^3*z^3-432*y^2*z^3-432*y*z^3-432*z^3+264*y^14*z^2
           +264*y^13*z^2+264*y^12*z^2-1240*y^11*z^2-1240*y^10*z^2-1240*y^9*z^2
           +1080*y^8*z^2+1872*y^7*z^2+1872*y^6*z^2-288*y^5*z^2-288*y^4*z^2
           -1080*y^3*z^2-80*y^2*z^2-80*y*z^2-80*z^2+96*y^14*z+96*y^13*z
           +96*y^12*z-480*y^11*z-480*y^10*z-480*y^9*z+480*y^8*z+768*y^7*z
           +768*y^6*z-192*y^5*z-192*y^4*z-480*y^3*z+16*y^14+16*y^13+16*y^12
           -80*y^11-80*y^10-80*y^9+80*y^8+128*y^7+128*y^6-32*y^5-32*y^4
           -80*y^3,
       (-62093*z^21)-62093*x*z^20-294905*z^20-124186*y^3*z^19+372558*y*z^19
                    -481184*x*z^19-1896895*z^19+62093*y^4*z^18-93066*y^3*z^18
                    +620930*y^2*z^18+775942*y*z^18-1974401*x*z^18-5672158*z^18
                    -62093*y^6*z^17+2157695*y^4*z^17-2614317*y^3*z^17
                    +1955562*y^2*z^17-1285014*y*z^17-4548614*x*z^17
                    -3792326*z^17+124186*y^7*z^16+1071141*y^6*z^16
                    +2732092*y^5*z^16+7080558*y^4*z^16-14439323*y^3*z^16
                    -279292*y^2*z^16-4633130*y*z^16-5402786*x*z^16
                    +17358124*z^16-62093*y^8*z^15+3818646*y^7*z^15
                    +1051714*y^6*z^15+10988844*y^5*z^15+6495674*y^4*z^15
                    -20470190*y^3*z^15-9851406*y^2*z^15-138226*y*z^15
                    +1471927*x*z^15+37860730*z^15+62093*y^10*z^14
                    +1241860*y^9*z^14+4051605*y^8*z^14+14285596*y^7*z^14
                    -10057949*y^6*z^14+12430164*y^5*z^14-1846064*y^4*z^14
                    +20484593*y^3*z^14-16124282*y^2*z^14+5184446*y*z^14
                    +15375648*x*z^14+4109437*z^14-124186*y^11*z^13
                    +2654439*y^10*z^13+3166008*y^9*z^13+18125525*y^8*z^13
                    +19298896*y^7*z^13-21572134*y^6*z^13-10283576*y^5*z^13
                    +885080*y^4*z^13+83602226*y^3*z^13-8540932*y^2*z^13
                    -13306902*y*z^13+22142511*x*z^13-75897065*z^13
                    +372558*y^12*z^12+2390654*y^11*z^12+10433286*y^10*z^12
                    +994182*y^9*z^12+28414036*y^8*z^12+7117194*y^7*z^12
                    +6413327*y^6*z^12-40089496*y^5*z^12+1387642*y^4*z^12
                    +51887995*y^3*z^12+3321430*y^2*z^12-34770050*y*z^12
                    +5326449*x*z^12-89966545*z^12-62093*y^14*z^11
                    +620930*y^13*z^11+1148500*y^12*z^11+11665603*y^11*z^11
                    +18068852*y^10*z^11-1766604*y^9*z^11+9581124*y^8*z^11
                    -9270810*y^7*z^11+59209974*y^6*z^11-50953520*y^5*z^11
                    -49868373*y^4*z^11-93415586*y^3*z^11+20070502*y^2*z^11
                    +3870366*y*z^11-20991981*x*z^11+21700007*z^11
                    +450211*y^14*z^10+2576492*y^13*z^10+1726276*y^12*z^10
                    +20826185*y^11*z^10+18157977*y^10*z^10+10234776*y^9*z^10
                    -32226896*y^8*z^10-26804700*y^7*z^10+39254296*y^6*z^10
                    -45553956*y^5*z^10-105032993*y^4*z^10-161017882*y^3*z^10
                    +51189864*y^2*z^10+77880544*y*z^10-25192515*x*z^10
                    +135785208*z^10+2449174*y^14*z^9+5401850*y^13*z^9
                    +2463088*y^12*z^9+14515838*y^11*z^9+10384929*y^10*z^9
                    +30607428*y^9*z^9-69123395*y^8*z^9-66254262*y^7*z^9
                    -78227581*y^6*z^9-701820*y^5*z^9-31820374*y^4*z^9
                    -13363721*y^3*z^9+50901220*y^2*z^9+72922124*y*z^9
                    -3694290*x*z^9+98509917*z^9+5152725*y^14*z^8
                    +7811974*y^13*z^8+5337046*y^12*z^8-9842841*y^11*z^8
                    -8576554*y^10*z^8+19746518*y^9*z^8-81091480*y^8*z^8
                    -84309180*y^7*z^8-143215725*y^6*z^8+101520364*y^5*z^8
                    +129546847*y^4*z^8+170255391*y^3*z^8-21177336*y^2*z^8
                    -25219024*y*z^8+10699009*x*z^8-39012662*z^8
                    +5893152*y^14*z^7+8669290*y^13*z^7+9191446*y^12*z^7
                    -39441258*y^11*z^7-43983913*y^10*z^7-35115338*y^9*z^7
                    -35307862*y^8*z^7-4494634*y^7*z^7-32549470*y^6*z^7
                    +142704156*y^5*z^7+167383594*y^4*z^7+137957690*y^3*z^7
                    -81223016*y^2*z^7-88829334*y*z^7+7091971*x*z^7
                    -103259594*z^7+2999595*y^14*z^6+5973124*y^13*z^6
                    +8486896*y^12*z^6-55471863*y^11*z^6-68351032*y^10*z^6
                    -78724496*y^9*z^6+66547639*y^8*z^6+118530918*y^7*z^6
                    +132719201*y^6*z^6+28845308*y^5*z^6+28654796*y^4*z^6
                    -45834331*y^3*z^6-46221546*y^2*z^6-47570630*y*z^6
                    +7776*x^2*z^6+217021*x*z^6-50374091*z^6-3276945*y^14*z^5
                    -1522340*y^13*z^5+394832*y^12*z^5-35486271*y^11*z^5
                    -43934307*y^10*z^5-53390064*y^9*z^5+115182683*y^8*z^5
                    +133783318*y^7*z^5+148478637*y^6*z^5-90595828*y^5*z^5
                    -93073209*y^4*z^5-128446523*y^3*z^5+17771150*y^2*z^5
                    +16921524*y*z^5+15552*x^2*z^5+7776*x*z^5+17195287*z^5
                    -9750375*y^14*z^4-9389888*y^13*z^4-9020896*y^12*z^4
                    +15841297*y^11*z^4+13952111*y^10*z^4+12147732*y^9*z^4
                    +48088910*y^8*z^4+26287984*y^7*z^4+29119635*y^6*z^4
                    -72824464*y^5*z^4-73097103*y^4*z^4-55627499*y^3*z^4
                    +28155166*y^2*z^4+28220776*y*z^4+23328*x^2*z^4+9720*x*z^4
                    +27941326*z^4-11437557*y^14*z^3-11328236*y^13*z^3
                    -11280880*y^12*z^3+47780995*y^11*z^3+47175827*y^10*z^3
                    +46880484*y^9*z^3-29152980*y^8*z^3-62322100*y^7*z^3
                    -61589201*y^6*z^3-4522320*y^5*z^3-5850743*y^4*z^3
                    +27747047*y^3*z^3+8893592*y^2*z^3+9356264*y*z^3
                    -23328*x^2*z^3+5832*x*z^3+9673136*z^3-6849778*y^14*z^2
                    -6905668*y^13*z^2-6894490*y^12*z^2+33943290*y^11*z^2
                    +34193580*y^10*z^2+34174626*y^9*z^2-33299528*y^8*z^2
                    -54131228*y^7*z^2-54345068*y^6*z^2+12452276*y^5*z^2
                    +13044224*y^4*z^2+33736442*y^3*z^2+538394*y^2*z^2
                    +247766*y*z^2-23328*x^2*z^2+95162*z^2-1891498*y^14*z
                    -1884694*y^13*z-1910938*y^12*z+9409862*y^11*z
                    +9408890*y^10*z+9573158*y^9*z-9206714*y^8*z-15072692*y^7*z
                    -15315692*y^6*z+3496256*y^5*z+3690656*y^4*z+54432*x*y^3*z
                    +9648974*y^3*z-93312*x*y^2*z+93312*y^2*z+7776*x^2*y*z
                    -7776*y*z-7776*x^2*z+38880*x*z-31104*z-67438*y^14
                    -78130*y^13-55774*y^12+373154*y^11+393566*y^10+248738*y^9
                    -541310*y^8-571580*y^7-359684*y^6+386624*y^5+200000*y^4
                    -54432*x*y^3+126266*y^3+93312*x*y^2-93312*y^2-7776*x^2*y
                    +7776*y+7776*x^2-38880*x+31104,
       62197*z^21+62197*x*z^20+291088*z^20+124394*y^3*z^19-373182*y*z^19
                 +477679*x*z^19+1878962*z^19-62197*y^4*z^18+84600*y^3*z^18
                 -621970*y^2*z^18-751376*y*z^18+1943665*x*z^18+5547434*z^18
                 +62197*y^6*z^17-2156998*y^4*z^17+2610963*y^3*z^17
                 -1915728*y^2*z^17+1344852*y*z^17+4414720*x*z^17+3387640*z^17
                 -124394*y^7*z^16-1077246*y^6*z^16-2736668*y^5*z^16
                 -6941979*y^4*z^16+14282032*y^3*z^16+421886*y^2*z^16
                 +4556674*y*z^16+5079481*x*z^16-17694533*z^16+62197*y^8*z^15
                 -3816420*y^7*z^15-979745*y^6*z^15-10817568*y^5*z^15
                 -5993371*y^4*z^15+19475671*y^3*z^15+9863628*y^2*z^15
                 -201616*y*z^15-1881863*x*z^15-36718052*z^15-62197*y^10*z^14
                 -1243940*y^9*z^14-4062702*y^8*z^14-14043134*y^7*z^14
                 +10159270*y^6*z^14-11660088*y^5*z^14+2355712*y^4*z^14
                 -22068016*y^3*z^14+15450484*y^2*z^14-5236534*y*z^14
                 -15324612*x*z^14-1299572*z^14+124394*y^11*z^13
                 -2654574*y^10*z^13-3085092*y^9*z^13-17875228*y^8*z^13
                 -18301298*y^7*z^13+20912123*y^6*z^13+11254696*y^5*z^13
                 -998824*y^4*z^13-82420621*y^3*z^13+7344470*y^2*z^13
                 +13720314*y*z^13-21069324*x*z^13+76544146*z^13
                 -373182*y^12*z^12-2403280*y^11*z^12-10265835*y^10*z^12
                 -763326*y^9*z^12-27161252*y^8*z^12-5672886*y^7*z^12
                 -8030785*y^6*z^12+39489536*y^5*z^12-1373297*y^4*z^12
                 -45846113*y^3*z^12-4006574*y^2*z^12+33940378*y*z^12
                 -3669771*x*z^12+84649877*z^12+62197*y^14*z^11
                 -621970*y^13*z^11-1124558*y^12*z^11-11520437*y^11*z^11
                 -17348077*y^10*z^11+1861182*y^9*z^11-7471074*y^8*z^11
                 +9874692*y^7*z^11-58997319*y^6*z^11+48089980*y^5*z^11
                 +50094117*y^4*z^11+97789063*y^3*z^11-19868540*y^2*z^11
                 -6467994*y*z^11+21506892*x*z^11-28651966*z^11
                 -455276*y^14*z^10-2537698*y^13*z^10-1645616*y^12*z^10
                 -20025700*y^11*z^10-16847943*y^10*z^10-10386480*y^9*z^10
                 +33102991*y^8*z^10+26175312*y^7*z^10-35006123*y^6*z^10
                 +41808924*y^5*z^10+101736412*y^4*z^10+154747355*y^3*z^10
                 -49836054*y^2*z^10-77962850*y*z^10+23712288*x*z^10
                 -134827431*z^10-2422655*y^14*z^9-5225662*y^13*z^9
                 -2338550*y^12*z^9-12994009*y^11*z^9-9032604*y^10*z^9
                 -29961138*y^9*z^9+66936577*y^8*z^9+64418844*y^7*z^9
                 +81555203*y^6*z^9-2708652*y^5*z^9+24063602*y^4*z^9
                 +1148833*y^3*z^9-47268530*y^2*z^9-67373776*y*z^9
                 +1764237*x*z^9-88593549*z^9-4986216*y^14*z^8-7428554*y^13*z^8
                 -5165504*y^12*z^8+10993230*y^11*z^8+9390554*y^10*z^8
                 -17534260*y^9*z^8+76120232*y^8*z^8+79656150*y^7*z^8
                 +137987784*y^6*z^8-101941496*y^5*z^8-132624275*y^4*z^8
                 -172225566*y^3*z^8+25125354*y^2*z^8+30962138*y*z^8
                 -11052938*x*z^8+46839364*z^8-5523915*y^14*z^7
                 -8105042*y^13*z^7-8820386*y^12*z^7+38847465*y^11*z^7
                 +43483352*y^10*z^7+36765466*y^9*z^7+29301146*y^8*z^7
                 -1847500*y^7*z^7+21773417*y^6*z^7-135684168*y^5*z^7
                 -158340584*y^4*z^7-125604427*y^3*z^7+80065726*y^2*z^7
                 +87379542*y*z^7-3888*x^2*z^7-6288353*x*z^7+100799218*z^7
                 -2561289*y^14*z^6-5335154*y^13*z^6-7823078*y^12*z^6
                 +52680669*y^11*z^6+65292989*y^10*z^6+76413538*y^9*z^6
                 -69509327*y^8*z^6-119457606*y^7*z^6-135926551*y^6*z^6
                 -18081196*y^5*z^6-15863623*y^4*z^6+56776217*y^3*z^6
                 +40208808*y^2*z^6+40930240*y*z^6-11664*x^2*z^6+343198*x*z^6
                 +42613627*z^6+3518784*y^14*z^5+1983340*y^13*z^5
                 +253010*y^12*z^5+31408200*y^11*z^5+38911488*y^10*z^5
                 +47606934*y^9*z^5-110691688*y^8*z^5-125367140*y^7*z^5
                 -139019436*y^6*z^5+93428348*y^5*z^5+95842332*y^4*z^5
                 +125597464*y^3*z^5-21532348*y^2*z^5-20745570*y*z^5
                 -19440*x^2*z^5-5832*x*z^5-21217046*z^5+9539685*y^14*z^4
                 +9316858*y^13*z^4+9093788*y^12*z^4-18625895*y^11*z^4
                 -17408485*y^10*z^4-16339548*y^9*z^4-39460090*y^8*z^4
                 -16066196*y^7*z^4-17749911*y^6*z^4+66181556*y^5*z^4
                 +66222615*y^4*z^4+45915757*y^3*z^4-26910314*y^2*z^4
                 -27005570*y*z^4-11664*x*z^4-26735354*z^4+10740936*y^14*z^3
                 +10658716*y^13*z^3+10641134*y^12*z^3-46816940*y^11*z^3
                 -46335370*y^10*z^3-46176990*y^9*z^3+33130656*y^8*z^3
                 +64709996*y^7*z^3+64170334*y^6*z^3-1268748*y^5*z^3
                 +131590*y^4*z^3-32235268*y^3*z^3-6653686*y^2*z^3
                 -7185370*y*z^3+19440*x^2*z^3-7526542*z^3+6000836*y^14*z^2
                 +6064016*y^13*z^2+6053324*y^12*z^2-30408756*y^11*z^2
                 -30703272*y^10*z^2-30698412*y^9*z^2+31174168*y^8*z^2
                 +49525624*y^7*z^2+49799728*y^6*z^2-12830488*y^5*z^2
                 -13526440*y^4*z^2-31703908*y^3*z^2+133388*y^2*z^2
                 +469700*y*z^2+15552*x^2*z^2+650492*z^2+1367102*y^14*z
                 +1360298*y^13*z+1388486*y^12*z-6785938*y^11*z-6786910*y^10*z
                 -6962842*y^9*z+6567238*y^8*z+10879468*y^7*z+11139964*y^6*z
                 -2410528*y^5*z-2641864*y^4*z-69984*x*y^3*z-7044490*y^3*z
                 +104976*x*y^2*z-104976*y^2*z-34992*x*z+34992*z-74500*y^14
                 -64780*y^13-90052*y^12+331676*y^11+318068*y^10+479420*y^9
                 -139220*y^8-559064*y^7-796232*y^6-148432*y^5+77072*y^4
                 +69984*x*y^3+596060*y^3-104976*x*y^2+104976*y^2+34992*x
                 -34992]" '(x y z)))))
    (&body)))

(test buchberger-cox-tough
  "Buchberger algorithm - Cox tough example"
  (with-fixture buchberger-cox-tough-context ()
    (is-true (grobner-test gb fl))
    ;;
    ;; TODO: The following test runs out of memory with normal strategy, as expected.
    ;;
    ;; NOTE: Cannot compare using SET-EXCLUSIVE-OR, as the Grobner basis
    ;; FL was computed using "sugar" strategy and is different from
    ;; the one obtained with BUCHBERGER, which uses the "normal" strategy.
    ;; (is (grobner-equal (buchberger fl) gb))
    ;;(is (every #'universal-equalp (parallel-buchberger fl) gb))
    )
  )


(run! 'buchberger-suite)
(format t "All tests done!~%")


