;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

(defpackage #:5am-monom
  (:use :cl :it.bese.fiveam :monom :copy)
  (:documentation "Monom and derived classes tests"))

(in-package :5am-monom)

(def-suite monom-suite 
    :description "Monom package suite")

(in-suite monom-suite)

(def-fixture monom-context ()
  ;; Use SYMBOL-MACROLET not let to avoid 'unused variable' complaints
  (symbol-macrolet 
      ((m (make-instance 'monom  :exponents '(1 2 3)))
       (n (make-instance 'monom  :exponents '(4 5 6)))
       (m*n (make-instance 'monom  :exponents '(5 7 9)))
       (n/m (make-instance 'monom  :exponents '(3 3 3)))
       (m-tensor-n (make-instance 'monom  :exponents '(1 2 3 4 5 6))))
    (&body)))

(test monom-basics
  "Monom basics"
  (with-fixture monom-context () 
    (is (= (monom-dimension m) 3))
    (is (= (monom-elt m 2) 3))
    (is (= (total-degree m) 6))
    (is (= (sugar m) 6))
    (is (equalp  (->list (make-instance 'monom  :dimension 3)) '(0 0 0)) "Trivial monomial is a vector of 0's")
    (is (universal-equalp (multiply m n) m*n))
    (is (universal-equalp (divide-by n m) n/m))
    (is (universal-equalp (right-tensor-product-by m n) m-tensor-n))
    (signals
	(error "EXPONENTS must have length DIMENSION") 
      (make-instance 'monom  :dimension 3 :exponents '(1 2 3 4 5 6)))
    (is-true (divides-p m n))
    (is-false (divides-p n m))
    (is (universal-equalp (universal-gcd m n) m))
    (is (universal-equalp (universal-lcm m n) n))
    (is-true (depends-p m 0))
    (signals
	(error "Index out of bounds")
      (depends-p m 3))
    )
  (with-fixture monom-context ()
    (is (universal-equalp (multiply-by m n) m*n)))
  (with-fixture monom-context ()
    (is (universal-equalp (divide-by n m) n/m)))
  (with-fixture monom-context () 
    (is (equal (->list m) '(1 2 3)))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        Order generics (LEX>, GRLEX>,...) tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-fixture order-context ()
  (symbol-macrolet 
      ((p (make-instance 'monom :exponents '(1 3 2)))
       (q (make-instance 'monom :exponents '(1 2 3))))
    (&body)))

(test order
  "order"
  (with-fixture order-context ()
    (is-true (lex>  p q)) 
    (is-true (grlex>  p q)) 
    (is-true (revlex>  p q)) 
    (is-true (grevlex>  p q)) 
    (is-false (invlex>  p q))))

(def-fixture elim-order-context ()
  (let* ((p (make-instance 'monom :exponents '(1 2 3)))
	 (q (make-instance 'monom :exponents '(4 5 6)))
	 (elim-order-factory (make-elimination-order-factory))
	 (elim-order-1 (funcall elim-order-factory 1))
	 (elim-order-2 (funcall elim-order-factory 2)))
    (&body)))


(test elim-order
  "Elimination order"
  (with-fixture elim-order-context ()
    (is-false (funcall elim-order-1 p q))
    (is-false (funcall elim-order-2 p q))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        TERM class tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-fixture term-context ()
  (symbol-macrolet 
      ((z (make-instance 'term  :dimension 3 :coeff 5))
       (m (make-instance 'term  :dimension 3 :exponents '(1 2 3) :coeff 6))
       (n (make-instance 'term  :dimension 3 :exponents '(4 5 6) :coeff 12))
       (m*n (make-instance 'term  :dimension 3 :exponents '(5 7 9) :coeff 72))
       (n/m (make-instance 'term  :dimension 3 :exponents '(3 3 3) :coeff 2))
       (m-tensor-n (make-instance 'term  :exponents '(1 2 3 4 5 6) :coeff 72))
       (m-uminus (make-instance 'term  :dimension 3 :exponents '(1 2 3) :coeff -6)))
    (&body)))

(test term-basics
  "Term basics"
  (with-fixture term-context () 
    (is (= (monom-dimension m) 3))
    (is (= (monom-elt m 2) 3))
    (is (= (total-degree m) 6))
    (is (= (sugar m) 6))
    (is (equalp  (->list z) '((0 0 0) . 5)) "Trivial term is a vector of 0's")
    (is (universal-equalp (multiply m n) m*n))
    (is (universal-equalp (divide n m) n/m))
    (is (universal-equalp (right-tensor-product-by m n) m-tensor-n))
    (signals
	(error "EXPONENTS must have length DIMENSION") 
      (make-instance 'term  :dimension 3 :exponents '(1 2 3 4 5 6) :coeff 77))
    (is-true (divides-p m n))
    (is-false (divides-p n m))
    (is (universal-equalp (universal-gcd m n) m))
    (is (universal-equalp (universal-lcm m n) n))
    (is-true (depends-p m 0))
    (signals
	(error "Index out of bounds")
      (depends-p m 3))
    )
  (with-fixture term-context ()
    (is (universal-equalp (multiply-by m n) m*n)))
  (with-fixture term-context ()
    (is (universal-equalp (divide-by n m) n/m)))
  (with-fixture term-context ()
    (is (universal-equalp (unary-minus m) m-uminus))))


(def-fixture monom/term-conversion-context ()
  (symbol-macrolet 
      ((term (make-instance 'term :exponents '(1 2 3) :coeff 4))
       (monom (make-instance 'monom :exponents '(1 2 3)))
       (promoted-monom  (make-instance 'term :exponents '(1 2 3) :coeff 1)))
    (&body)))

(test monom/term-conversion
  "Monom/term conversion"
  (with-fixture monom/term-conversion-context ()
    (is (universal-equalp (change-class term 'monom) monom)))
  (with-fixture monom/term-conversion-context ()
    (is (universal-equalp (change-class monom 'term) promoted-monom))))

(test monom/term-copy
  "Monom/term copy"
  (with-fixture monom/term-conversion-context ()
    (is (universal-equalp (copy-instance monom) monom))
    (is (universal-equalp (copy-instance term) term))))

(test term-tensor-product
  "Term tensor product"
  (let ((term1 (make-instance 'term :exponents '(1 2 3) :coeff 4))
	(term2 (make-instance 'term :exponents '(4 5) :coeff 3))
	(term1-left-tensor-by-term2 (make-instance 'term :exponents '(4 5 1 2 3) :coeff 12)))
    (is (universal-equalp (left-tensor-product-by term1 term2) term1-left-tensor-by-term2))))

(test term-contract
  "Term contract"
  (let ((term (make-instance 'term :exponents '(1 2 3) :coeff 4))
	(term-contracted (make-instance 'term :exponents '(2 3) :coeff 4)))
    (is (universal-equalp (left-contract term 1) term-contracted))))

(test sexp-conversion
  "Sexp conversion"
  (is (equal (->sexp (make-instance 'term :exponents '(0 0 0 0) :coeff -5) '(x y u v)) -5))
  (signals (error "Variables NIL and exponents #(0 0 0 0) must have the same length.")
    (->sexp (make-instance 'monom :exponents '(0 0 0 0))))
  (is (equal (->sexp (make-instance 'monom :exponents '(0 0 0 0)) '(x y u w)) 1))
  (is (equal (->sexp (make-instance 'monom :exponents '(0 0 1 0)) '(x y u w)) 'u))
  (is (equalp (->sexp (make-instance 'monom :exponents '(1 2 1 1)) '(x y u w))
	      '(* X (EXPT Y 2) U W)))
  (is (equal (->sexp (make-instance 'term :exponents '(0 1 0 0) :coeff -5) '(x y u v))
	     '(* -5 Y)))
  (is (equal (->sexp (make-instance 'term :exponents '(1 1 0 0) :coeff -5) '(x y u v))
	     '(* -5 X Y)))
  (is (equal (->sexp (make-instance 'term :exponents '(1 0 0 0) :coeff -5) '(x y u v))
	     '(* -5 X)))
  (is (equal (->sexp (make-instance 'term :exponents '(11 0 0 0) :coeff -5) '(x y u v))
	     '(* -5 (EXPT X 11))))
  (is (equal (->sexp (make-instance 'term :exponents '(11 0 0 0) :coeff 5) '(x y u v))
	     '(* 5 (EXPT X 11))))
  (is (equal (->sexp (make-instance 'term :exponents '(11 0 4 0) :coeff 5) '(x y u v))
	     '(* 5 (EXPT X 11) (EXPT U 4))))
)

(run! 'monom-suite)
(format t "All tests done!~%")


