;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

;; Unless NGROBNER system loaded by ASDF,
;; load the dependencies directly
#-ngrobner
(progn
  (require :utils "utils")
  (require :copy "copy")
  (require :monom "monom")
  (require :polynomial "polynomial")
  (require :infix "infix")
  (require :symbolic-polynomial "symbolic-polynomial"))

(defpackage #:5am-symbolic-poly
  (:use :cl :it.bese.fiveam :monom :polynomial :infix :symbolic-polynomial))

(in-package :5am-symbolic-poly)

(def-suite symbolic-poly-suite 
    :description "Symbolic polynomial package suite")

(in-suite symbolic-poly-suite)

(test read-infix-form
  "Infix form reader"
  (is (equalp (with-input-from-string (s "X^2-Y^2+(-4/3)*U^2*W^3-5") (read-infix-form :stream s))
	      '(+ (- (EXPT X 2) (EXPT Y 2)) (* (- (/ 4 3)) (EXPT U 2) (EXPT W 3)) (- 5))))
  (is (equalp (poly->string (string->poly "1" '(x y u w))) "1"))
  (is (equalp (->sexp (string->poly "X^2-Y^2+(-4/3)*U^2*W^3-5" '(x y u w)))
	      '(+ (EXPT X 2) (* -1 (EXPT Y 2)) (* -4/3 (EXPT U 2) (EXPT W 3)) -5))))


(def-fixture sym-poly-context ()
  (let ((p (make-instance 'poly))
	(p-symbolic (make-instance 'symbolic-poly :vars '(x))))
    (dolist (x '( ((2) . 22)  ((4) . 44) ((5) . 55) ((8) . 88) ((9) . 99) ))
      (poly-insert-term p (make-instance 'term :exponents (car x) :coeff (cdr x)))
      (poly-insert-term p-symbolic (make-instance 'term :exponents (car x) :coeff (cdr x))))
    (&body)))

(test sym-poly
  "Symbolic polynomial"
  (with-fixture sym-poly-context ()
    (is (universal-equalp (change-class p 'symbolic-poly :vars '(x)) p-symbolic )))
  (with-fixture sym-poly-context ()
    (signals
	(error "Number of variables does not equal dimension.")
      (universal-equalp (change-class p 'symbolic-poly :vars '(x y)) p-symbolic ))))

;; The following example uses the Grobner basis obtained with Maxima
;; Grobner package
;;
;; poly_grobner([x-3*u-3*u*v^2+u^3,y-3*v-3*u^2*v+v^3,z-3*u^2+3*v^2],[u,v,x,y,z]);
;;
;; The Maxima output is placed in the string STR-1.  The string STR-1
;; is parsed STRING->POLY to a polynomial list GB-1.  Then we take GB-1 and apply
;; POLY->STRING to it. Then we feed the string STR-2 into Maxima as
;; input and format according to Maxima rules. The resulting string is
;; STR-3. We parse it with STRING->POLY to a polynomial list GB-2.
;; Obviously, we should have equality GB-1 and GB-2. This is the test.
;;
(def-fixture big-poly-list-context ()
  (let* ((vars '(u v x y z))
	 (str-1 "[x-3*u*v^2+u^3-3*u,y+v^3-3*u^2*v-3*v,z+3*v^2-3*u^2,
       (-u*z)-3*x+6*u*v^2+9*u,(-v*z)+y-2*v^3-3*v,z^2+6*v^2*z-9*z-9*v*y+9*u*x,
       4*u*v*z-3*u*y+3*v*x,2*u*z^2+6*x*z-18*u*z-9*u*v*y+9*v^2*x,
       (-8*u*z^3)-24*x*z^2+72*u*z^2-36*v^2*x*z+27*u*y^2-27*v*x*y,
       z^3+18*v^2*z^2-18*z^2-54*v*y*z+54*v^2*z+81*z+27*y^2-27*x^2,
       (-4*z^4)+48*z^3-108*v*y*z^2+108*z^2+135*y^2*z+324*v*y*z+108*x^2*z
               -1296*v^2*z-1944*z-243*v^2*y^2-648*y^2+243*v^2*x^2+648*x^2,
       8*v*z^3-9*y*z^2+72*v*z^2+54*v^2*y*z-27*y*z-27*v*y^2+27*v*x^2,
       (-8*v*z^4)+12*y*z^3-96*v*z^3-216*v*z^2-135*v*y^2*z+324*y*z-27*v*x^2*z
                 +81*y^3+81*v*y^2-81*x^2*y-81*v*x^2,
       (-64*v*z^6)+120*y*z^5-1152*v*z^5+288*y*z^4-5184*v*z^4-648*v*y^2*z^3
                  -216*y*z^3+6912*v*z^3+81*y^3*z^2-9720*v*y^2*z^2
                  -1539*x^2*y*z^2+31104*y*z^2+62208*v*z^2+8505*y^3*z
                  +46656*v*y^2*z-8505*x^2*y*z-93312*y*z+729*v*y^4-23328*y^3
                  -1458*v*x^2*y^2-23328*v*y^2+23328*x^2*y+729*v*x^4
                  +23328*v*x^2,
       8*z^6-72*z^5+648*v*y*z^4-648*z^4-945*y^2*z^3+5184*v*y*z^3-189*x^2*z^3
            +5832*z^3+972*y^2*z^2+17496*v*y*z^2-2430*x^2*z^2+8748*v*y^3*z
            -19683*y^2*z+2187*x^2*z-5103*y^4-4374*v*y^3+5832*x^2*y^2
            +4374*v*x^2*y-729*x^4,
       8*z^7-48*z^6+648*v*y*z^5-864*z^5-945*y^2*z^4+5832*v*y*z^4-189*x^2*z^4
            +3888*z^4+81*y^2*z^3+17496*v*y*z^3-2997*x^2*z^3+17496*z^3
            +8748*v*y^3*z^2-16767*y^2*z^2+17496*v*y*z^2-5103*x^2*z^2
            -5103*y^4*z+5832*x^2*y^2*z-6561*y^2*z-729*x^4*z+6561*x^2*z
            -2187*y^4+4374*x^2*y^2-2187*x^4,
       64*z^9-10368*z^7+1296*y^2*z^6-1296*x^2*z^6-34992*y^2*z^5-34992*x^2*z^5
             +419904*z^5+174960*y^2*z^4-174960*x^2*z^4-10935*y^4*z^3
             -56862*x^2*y^2*z^3+314928*y^2*z^3-10935*x^4*z^3+314928*x^2*z^3
             +118098*y^4*z^2-118098*x^4*z^2+59049*y^4*z-118098*x^2*y^2*z
             +59049*x^4*z+19683*y^6-59049*x^2*y^4+59049*x^4*y^2-19683*x^6]")
	 (gb-1 (string->poly str-1 vars))
	 (str-2 (poly->string gb-1))
	 ;; str-3 is a Maxima-postprocessed str-2"
	 (str-3 "[X-3*U*V^2+U^3-3*U,Y+V^3-3*U^2*V-3*V,Z+3*V^2-3*U^2,
        (-U*Z)-3*X+6*U*V^2+9*U,(-V*Z)+Y-2*V^3-3*V,Z^2+6*V^2*Z-9*Z-9*V*Y+9*U*X,
        4*U*V*Z-3*U*Y+3*V*X,2*U*Z^2+6*X*Z-18*U*Z-9*U*V*Y+9*V^2*X,
        (-8*U*Z^3)-24*X*Z^2+72*U*Z^2-36*V^2*X*Z+27*U*Y^2-27*V*X*Y,
        Z^3+18*V^2*Z^2-18*Z^2-54*V*Y*Z+54*V^2*Z+81*Z+27*Y^2-27*X^2,
        (-4*Z^4)+48*Z^3-108*V*Y*Z^2+108*Z^2+135*Y^2*Z+324*V*Y*Z+108*X^2*Z
                -1296*V^2*Z-1944*Z-243*V^2*Y^2-648*Y^2+243*V^2*X^2+648*X^2,
        8*V*Z^3-9*Y*Z^2+72*V*Z^2+54*V^2*Y*Z-27*Y*Z-27*V*Y^2+27*V*X^2,
        (-8*V*Z^4)+12*Y*Z^3-96*V*Z^3-216*V*Z^2-135*V*Y^2*Z+324*Y*Z-27*V*X^2*Z
                  +81*Y^3+81*V*Y^2-81*X^2*Y-81*V*X^2,
        (-64*V*Z^6)+120*Y*Z^5-1152*V*Z^5+288*Y*Z^4-5184*V*Z^4-648*V*Y^2*Z^3
                   -216*Y*Z^3+6912*V*Z^3+81*Y^3*Z^2-9720*V*Y^2*Z^2
                   -1539*X^2*Y*Z^2+31104*Y*Z^2+62208*V*Z^2+8505*Y^3*Z
                   +46656*V*Y^2*Z-8505*X^2*Y*Z-93312*Y*Z+729*V*Y^4-23328*Y^3
                   -1458*V*X^2*Y^2-23328*V*Y^2+23328*X^2*Y+729*V*X^4
                   +23328*V*X^2,
        8*Z^6-72*Z^5+648*V*Y*Z^4-648*Z^4-945*Y^2*Z^3+5184*V*Y*Z^3-189*X^2*Z^3
             +5832*Z^3+972*Y^2*Z^2+17496*V*Y*Z^2-2430*X^2*Z^2+8748*V*Y^3*Z
             -19683*Y^2*Z+2187*X^2*Z-5103*Y^4-4374*V*Y^3+5832*X^2*Y^2
             +4374*V*X^2*Y-729*X^4,
        8*Z^7-48*Z^6+648*V*Y*Z^5-864*Z^5-945*Y^2*Z^4+5832*V*Y*Z^4-189*X^2*Z^4
             +3888*Z^4+81*Y^2*Z^3+17496*V*Y*Z^3-2997*X^2*Z^3+17496*Z^3
             +8748*V*Y^3*Z^2-16767*Y^2*Z^2+17496*V*Y*Z^2-5103*X^2*Z^2
             -5103*Y^4*Z+5832*X^2*Y^2*Z-6561*Y^2*Z-729*X^4*Z+6561*X^2*Z
             -2187*Y^4+4374*X^2*Y^2-2187*X^4,
        64*Z^9-10368*Z^7+1296*Y^2*Z^6-1296*X^2*Z^6-34992*Y^2*Z^5-34992*X^2*Z^5
              +419904*Z^5+174960*Y^2*Z^4-174960*X^2*Z^4-10935*Y^4*Z^3
              -56862*X^2*Y^2*Z^3+314928*Y^2*Z^3-10935*X^4*Z^3+314928*X^2*Z^3
              +118098*Y^4*Z^2-118098*X^4*Z^2+59049*Y^4*Z-118098*X^2*Y^2*Z
              +59049*X^4*Z+19683*Y^6-59049*X^2*Y^4+59049*X^4*Y^2-19683*X^6]")
	 (gb-2 (string->poly str-3 vars)))
    (declare (ignore str-2))
    (&body)))

(test big-poly-list
  "Big polynomial list"
  (with-fixture big-poly-list-context ()
    (is (universal-equalp gb-1 gb-2))))
    

(run! 'symbolic-poly-suite)
(format t "All tests done!~%")


