;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "PAIR-QUEUE"
  (:use :cl :priority-queue :monom :polynomial :utils)
  (:export "PAIR"
	   "MAKE-PAIR"
	   "PAIR-FIRST"
	   "PAIR-SECOND"
	   "PAIR-DIVISION-DATA"
	   "*PAIR-KEY-FUNCTION*"
	   "*PAIR-ORDER*"
	   "MAKE-PAIR-QUEUE"
	   "PAIR-QUEUE-INITIALIZE"
	   "PAIR-QUEUE-INSERT"
	   "PAIR-QUEUE-REMOVE"
	   "PAIR-QUEUE-SIZE"
	   "PAIR-QUEUE-EMPTY-P"
	   "SET-PAIR-HEURISTIC"
	   )
  (:documentation "Critical pair queue implementation. The pair queue is a list of critical
pairs, ordered by some partial order. Pair queue is a kind of priority queue.")
  )
	   

(in-package :pair-queue)

(defclass critical-pair ()
  ((first :initform nil :initarg :first)
   (second :initform nil :initarg :second))
  (:documentation "Represents a critical pair."))

  
(defvar *pair-key-function* nil
  "Function that, given two polynomials as argument, computed the key
in the pair queue.")

(defvar *pair-order* nil
  "Function that orders the keys of pairs.")

(defun make-pair-queue ()
  "Constructs a priority queue for critical pairs."
  (make-priority-queue
   :element-type 'pair
   :element-key #'(lambda (pair) (funcall *pair-key-function* (pair-first pair) (pair-second pair)))
   :test *pair-order*))

(defun pair-queue-initialize (pq f start
			      &aux
			      (s (1- (length f)))
			      (b (nconc (makelist (make-pair (elt f i) (elt f j))
						 (i 0 (1- start)) (j start s))
					(makelist (make-pair (elt f i) (elt f j))
						 (i start (1- s)) (j (1+ i) s)))))
  "Initializes the priority for critical pairs. F is the initial list of polynomials.
START is the first position beyond the elements which form a partial
grobner basis, i.e. satisfy the Buchberger criterion."
  (declare (type priority-queue pq) (type fixnum start))
  (dolist (pair b pq)
    (priority-queue-insert pq pair)))

(defun pair-queue-insert (b pair)
  (priority-queue-insert b pair))

(defun pair-queue-remove (b)
  (priority-queue-remove b))

(defun pair-queue-size (b)
  (priority-queue-size b))

(defun pair-queue-empty-p (b)
  (priority-queue-empty-p b))

(defun set-pair-heuristic (method)
  "Sets up variables *PAIR-KEY-FUNCTION* and *PAIR-ORDER* used
to determine the priority of critical pairs in the priority queue."
  (ecase method
    ((minimal-lcm :minimal-lcm $minimal_lcm)
     (setf *pair-key-function* #'(lambda (p q)
				   (universal-lcm (leading-monomial p) (leading-monomial q)))
	   *pair-order* #'reverse-monomial-order))
    ((minimal-total-degree :minimal-total-degree $minimal_total_degree)
     (setf *pair-key-function* #'(lambda (p q)
				   (total-degree
				    (universal-lcm (leading-monomial p) (leading-monomial q))))
	   *pair-order* #'<))
    ((minimal-length :minimal-length $minimal_length)
     (setf *pair-key-function* #'(lambda (p q)
				   (+ (poly-length p) (poly-length q)))
	   *pair-order* #'<))))

