;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "TERM"
  (:use :cl :monom :ring)
  (:export "TERM" 
	   "MAKE-TERM-VARIABLE"
	   )
  (:documentation "This package implements class TERM. A term is a
product of a scalar and powers of some variables, such as
5*X^2*Y^3. The part of the term without the coefficient is a monomial
X^2*Y^3, which is represented by class MONOM, provided by the :MONOM
package. In this implementation, a TERM specializes MONOL. Also, a
monomial can be considered a TERM whose coefficient is the unit
element (1) of the underlying ring. The generic method CHANGE-CLASS
can be used to convert between a MONOM and a TERM, observing this
convention."))

(in-package :term)

(proclaim '(optimize (speed 3) (space 0) (safety 0) (debug 0)))

(defclass term (monom scalar)
  ()
  (:documentation "Implements a term, i.e. a product of a scalar
and powers of some variables, such as 5*X^2*Y^3."))

(defmethod print-object ((self term) stream)
  (with-accessors ((dimension monom-dimension)
		   (exponents monom-exponents)
		   (coeff scalar-coeff))
      self
    (format stream "#<TERM DIMENSION=~A EXPONENTS=~A COEFF=~A>"
	    dimension
	    exponents
	    coeff)))
    

(defmethod r-equalp ((term1 term) (term2 term))
  (and (r-equalp (scalar-coeff term1) (scalar-coeff term2))
       (= (monom-dimension term1) (monom-dimension term2))
       (equalp (monom-exponents term1) (monom-exponents term2))))


#|
(defmethod shared-initialize ((self term) slot-names
			      &rest 
				initargs 
			      &key
				coeff
				&allow-other-keys)
  (declare (ignore initargs))
  (if (eq slot-names t) (setf slot-names '(coeff)))
  (dolist (slot-name slot-names)
    (case slot-name
      (coeff
       (setf (slot-value self 'coeff) coeff)))))
|#

#|
(defmethod update-instance-for-different-class :after ((old monom) (new term) 
						&rest 
						  initargs 
						&key 
						  &allow-other-keys)
  (format t "Initargs:~A" initargs)
  (prog1 (call-next-method)
    (format t "Old:~A New:~A" old new))
  )
|#


#|
(defun make-term-variable (nvars pos
			   &optional
			     (power 1)
			     (coeff 1))
  "Construct a term in the polynomial ring RING[X[0],X[1],X[2],...X[NVARS-1]]
over the ring RING which represents a single variable. It assumes
number of variables NVARS and the variable is at position
POS. Optionally, the variable may appear raised to power POWER.
Optionally, the term may appear with an arbitrary coefficient, which
defaults to the unit of the RING."
  (declare (type fixnum nvars pos))
  (make-term :monom (make-monom-variable nvars pos power)
	     :coeff coeff))

|#

(defmethod multiply-by :before ((self term) (other term))
  "Destructively multiply terms SELF and OTHER and store the result into SELF.
It returns SELF."
  (setf (scalar-coeff self) (multiply-by (scalar-coeff self) (scalar-coeff other))))

(defmethod left-tensor-product-by ((self term) (other term))
  (setf (scalar-coeff self) (multiply-by (scalar-coeff self) (scalar-coeff other)))
  (call-next-method))

(defmethod right-tensor-product-by ((self term) (other term))
  (setf (scalar-coeff self) (multiply-by (scalar-coeff self) (scalar-coeff other)))
  (call-next-method))

(defmethod left-tensor-product-by ((self term) (other monom))
  (call-next-method))

(defmethod right-tensor-product-by ((self term) (other monom))
  (call-next-method))

(defmethod divide-by ((self term) (other term))
  "Destructively divide term SELF by OTHER and store the result into SELF.
It returns SELF."
  (setf (scalar-coeff self) (divide-by (scalar-coeff self) (scalar-coeff other)))
  (call-next-method))

(defmethod unary-minus ((self term))
  (setf (scalar-coeff self) (unary-minus (scalar-coeff self)))
  self)

(defmethod r* ((term1 term) (term2 term))
  "Non-destructively multiply TERM1 by TERM2."
  (multiply-by (copy-instance term1) (copy-instance term2)))

(defmethod r-zerop ((self term))
  (r-zerop (scalar-coeff self)))

#|

(defun term->cons (term) 
  "A human-readable representation of a term as a cons (MONOM . COEFF)."
  (declare (type term term))
  (cons (monom->list (term-monom term)) (scalar-coeff term)))

|#
