;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Run tests using 5am unit testing framework
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We assume that QuickLisp package manager is installed.
;; See :
;;      https://www.quicklisp.org/beta/
;;

;; The following is unnecessary after running:
;; * (ql:add-to-init-file)
;; at lisp prompt:
;;(load "~/quicklisp/setup")

(ql:quickload :fiveam)

(require :ring "ring")
(require :monom "monom")
(require :term "term")
(require :order "order")
(require :polynomial "polynomial")

(defpackage #:5am-monom
  (:use :cl :it.bese.fiveam :ring :monom :term :order :polynomial)
  (:shadowing-import-from :ring "ZEROP" "LCM" "GCD" "+" "-" "*" "/" "EXPT"))

(in-package :5am-monom)

(def-suite monom-suite 
    :description "Monom package suite")

(in-suite monom-suite)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        MONOM class tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-fixture monom-context ()
  (let ((z (make-instance 'monom  :dimension 3))
	(m (make-instance 'monom  :dimension 3 :exponents '(1 2 3)))
	(n (make-instance 'monom  :dimension 3 :exponents '(4 5 6)))
	(m*n (make-instance 'monom  :dimension 3 :exponents '(5 7 9)))
	(n/m (make-instance 'monom  :dimension 3 :exponents '(3 3 3)))
	(m-tensor-n (make-instance 'monom  :exponents '(1 2 3 4 5 6))))
    (&body)))

(test monom-basics
  "Monom basics"
  (with-fixture monom-context () 
    (is (= (r-dimension m) 3))
    (is (= (r-elt m 2) 3))
    (is (= (r-total-degree m) 6))
    (is (= (r-sugar m) 6))
    (is (equalp  (r->list z) '(0 0 0)) "Trivial monomial is a vector of 0's")
    (is (r-equalp (r* m n) m*n))
    (is (r-equalp (r/ n m) n/m))
    (is (r-equalp (r-tensor-product m n) m-tensor-n))
    (signals
	(error "EXPONENTS must have length DIMENSION") 
      (make-instance 'monom  :dimension 3 :exponents '(1 2 3 4 5 6)))
    (is-true (r-divides-p m n))
    (is-false (r-divides-p n m))
    (is (r-equalp (r-gcd m n) m))
    (is (r-equalp (r-lcm m n) n))
    (is-true (r-depends-p m 0))
    (signals
	(error "Index out of bounds")
      (r-depends-p m 3))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        TERM class tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-fixture term-context ()
  (let ((z (make-instance 'term  :dimension 3 :coeff 5))
	(m (make-instance 'term  :dimension 3 :exponents '(1 2 3) :coeff 6))
	(n (make-instance 'term  :dimension 3 :exponents '(4 5 6) :coeff 12))
	(m*n (make-instance 'term  :dimension 3 :exponents '(5 7 9) :coeff 72))
	(n/m (make-instance 'term  :dimension 3 :exponents '(3 3 3) :coeff 2))
	(m-tensor-n (make-instance 'term  :exponents '(1 2 3 4 5 6) :coeff 72)))
    (&body)))

(test term-basics
  "Term basics"
  (with-fixture term-context () 
    (is (= (r-dimension m) 3))
    (is (= (r-elt m 2) 3))
    (is (= (r-total-degree m) 6))
    (is (= (r-sugar m) 6))
    (is (equalp  (r->list z) '(0 0 0)) "Trivial term is a vector of 0's")
    (is (r-equalp (r* m n) m*n))
    (is (r-equalp (r/ n m) n/m))
    (is (r-equalp (r-tensor-product m n) m-tensor-n))
    (signals
	(error "EXPONENTS must have length DIMENSION") 
      (make-instance 'term  :dimension 3 :exponents '(1 2 3 4 5 6) :coeff 77))
    (is-true (r-divides-p m n))
    (is-false (r-divides-p n m))
    (is (r-equalp (r-gcd m n) m))
    (is (r-equalp (r-lcm m n) n))
    (is-true (r-depends-p m 0))
    (signals
	(error "Index out of bounds")
      (r-depends-p m 3))
    )

  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        Order generics (LEX>, GRLEX>,...) tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(def-fixture order-context ()
  (let ((p (make-instance 'monom :exponents '(1 3 2)))
	(q (make-instance 'monom :exponents '(1 2 3))))
    (&body)))

(test order
  "order"
  (with-fixture order-context ()
    (is-true (lex>  p q)) 
    (is-true (grlex>  p q)) 
    (is-true (revlex>  p q)) 
    (is-true (grevlex>  p q)) 
    (is-false (invlex>  p q))))

(def-fixture elim-order-context ()
  (let* ((p (make-instance 'monom :exponents '(1 2 3)))
	 (q (make-instance 'monom :exponents '(4 5 6)))
	 (elim-order-factory (make-elimination-order-factory))
	 (elim-order-1 (funcall elim-order-factory 1))
	 (elim-order-2 (funcall elim-order-factory 2)))
    (&body)))


(test elim-order
  "Elimination order"
  (with-fixture elim-order-context ()
    (is-false (funcall elim-order-1 p q))
    (is-false (funcall elim-order-2 p q))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;        POLY class tests
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#|
(test poly
  "Polynomial"
  (let ((p (make-instance 'poly))
	(q (make-instance 'poly)))
    (dotimes (i 5) 
      (insert-item p (make-instance 'term :exponents (list (* 3 i)) :coeff (1+ (* 3 i))))
      (insert-item q (make-instance 'term :exponents (list (+ (* 3 i) 2)) :coeff (1- (* 3 i)))))
    (is-true (print p))
    (is-true (print q))
    (add-to p q)
    (is-true (print p))
  ))
|#

(defun foo ()
  (let ((p (make-instance 'poly))
	(q (make-instance 'poly)))
    (dotimes (i 5) 
      (insert-item p (make-instance 'term :exponents (list (* 3 i)) :coeff (1+ (* 3 i))))
      (insert-item q (make-instance 'term :exponents (list (+ (* 3 i) 2)) :coeff (1- (* 3 i)))))
    (print p)
    (print q)
    (add-to p q)
    (print p)
  ))



(run! 'monom-suite)
(format t "All tests done!~%")


