;;; -*-  Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                              
;;;  Copyright (C) 1999, 2002, 2009, 2015 Marek Rychlik <rychlik@u.arizona.edu>		 
;;;  		       								 
;;;  This program is free software; you can redistribute it and/or modify	 
;;;  it under the terms of the GNU General Public License as published by	 
;;;  the Free Software Foundation; either version 2 of the License, or		 
;;;  (at your option) any later version.					 
;;; 		       								 
;;;  This program is distributed in the hope that it will be useful,		 
;;;  but WITHOUT ANY WARRANTY; without even the implied warranty of		 
;;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 
;;;  GNU General Public License for more details.				 
;;; 		       								 
;;;  You should have received a copy of the GNU General Public License		 
;;;  along with this program; if not, write to the Free Software 		 
;;;  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	 
;;;										 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Polynomials
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defpackage "POLYNOMIAL"
  (:use :cl :ring :monom :order :term #| :infix |# )
  (:export "POLY" 
	   ))

(in-package :polynomial)

(proclaim '(optimize (speed 3) (space 0) (safety 0) (debug 0)))

#|
	     ;;
	     ;; BOA constructor, by default constructs zero polynomial
	     (:constructor make-poly-from-termlist (termlist &optional (sugar (termlist-sugar termlist))))
	     (:constructor make-poly-zero (&aux (termlist nil) (sugar -1)))
	     ;; Constructor of polynomials representing a variable
	     (:constructor make-poly-variable (ring nvars pos &optional (power 1)
					       &aux
					       (termlist (list
							  (make-term-variable ring nvars pos power)))
					       (sugar power)))
	     (:constructor poly-unit (ring dimension
					   &aux
					   (termlist (termlist-unit ring dimension))
					   (sugar 0))))

|#

(defclass poly ()
  ((termlist :initarg :termlist :accessor poly-termlist))
  (:default-initargs :termlist nil))

(defmethod print-object ((self poly) stream)
  (format stream "#<POLY TERMLIST=~A >" (poly-termlist self)))

(defgeneric insert-item (object item)
  (:method ((self poly) (item term))
    (push item (poly-termlist self))
    self))


(defgeneric append-item (object item)
  (:method ((self poly) (item term))
    (setf (cdr (last (poly-termlist self))) (list item))
    self))


#|


;; Leading term
(defgeneric leading-term (object)
  (:method ((self poly)) 
    (car (poly-termlist self))))

;; Second term
(defgeneric second-leading-term (object) 
  (:method ((self poly))
    (cadar (poly-termlist self))))

;; Leading coefficient
(defgeneric leading-coefficient (object)
  (:method ((self poly))
    (r-coeff (leading-term self))))

;; Second coefficient
(defgeneric second-leading-coefficient (object)
  (:method ((self poly)) 
    (r-coeff (second-leading-term self))))

;; Testing for a zero polynomial
(defmethod r-zerop ((self poly))
  (null (poly-termlist self)))

;; The number of terms
(defmethod r-length ((self poly))
  (length (poly-termlist self)))



(defgeneric multiply-by (self other)
  (:method ((self poly) (other scalar))
    (mapc #'(lambda (term) (multiply-by term other)) (poly-termlist self))
    self)
  (:method ((self poly) (other monom))
    (mapc #'(lambda (term) (multiply-by term monom)) (poly-termlist self))
    self))

(defgeneric add-to (self other)
  (:method ((self poly) (other poly))))

(defgeneric subtract-from (self other)
  (:method ((self poly) (other poly))))

(defmethod unary-uminus (self))

(defun poly-standard-extension (plist &aux (k (length plist)))
  "Calculate [U1*P1,U2*P2,...,UK*PK], where PLIST=[P1,P2,...,PK]."
  (declare (list plist) (fixnum k))
  (labels ((incf-power (g i)
	     (dolist (x (poly-termlist g))
	       (incf (monom-elt (term-monom x) i)))
	     (incf (poly-sugar g))))
    (setf plist (poly-list-add-variables plist k))
    (dotimes (i k plist)
      (incf-power (nth i plist) i))))

(defun saturation-extension (ring f plist 
			     &aux 
			       (k (length plist))
			       (d (monom-dimension (poly-lm (car plist))))
			       f-x plist-x)
  "Calculate [F, U1*P1-1,U2*P2-1,...,UK*PK-1], where PLIST=[P1,P2,...,PK]."
  (declare (type ring ring))
  (setf f-x (poly-list-add-variables f k)
	plist-x (mapcar #'(lambda (x)
			    (setf (poly-termlist x)
				  (nconc (poly-termlist x)
					 (list (make-term :monom (make-monom :dimension d)
							  :coeff (funcall (ring-uminus ring) 
									  (funcall (ring-unit ring)))))))
			    x)
			(poly-standard-extension plist)))
  (append f-x plist-x))


(defun polysaturation-extension (ring f plist 
				 &aux 
				   (k (length plist))
				   (d (+ k (monom-dimension (poly-lm (car plist)))))
				   ;; Add k variables to f
				   (f (poly-list-add-variables f k))
				   ;; Set PLIST to [U1*P1,U2*P2,...,UK*PK]
				   (plist (apply #'poly-append (poly-standard-extension plist))))
  "Calculate [F, U1*P1+U2*P2+...+UK*PK-1], where PLIST=[P1,P2,...,PK]. It destructively modifies F."
  ;; Add -1 as the last term
  (declare (type ring ring))
  (setf (cdr (last (poly-termlist plist)))
	(list (make-term :monom (make-monom :dimension d)
			 :coeff (funcall (ring-uminus ring) (funcall (ring-unit ring))))))
  (append f (list plist)))

(defun saturation-extension-1 (ring f p) 
  "Calculate [F, U*P-1]. It destructively modifies F."
  (declare (type ring ring))
  (polysaturation-extension ring f (list p)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Evaluation of polynomial (prefix) expressions
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun coerce-coeff (ring expr vars)
  "Coerce an element of the coefficient ring to a constant polynomial."
  ;; Modular arithmetic handler by rat
  (declare (type ring ring))
  (make-poly-from-termlist (list (make-term :monom (make-monom :dimension (length vars))
					    :coeff (funcall (ring-parse ring) expr)))
			   0))

(defun poly-eval (expr vars 
		  &optional 
		    (ring +ring-of-integers+) 
		    (order #'lex>)
		    (list-marker :[)
		  &aux 
		    (ring-and-order (make-ring-and-order :ring ring :order order)))
  "Evaluate Lisp form EXPR to a polynomial or a list of polynomials in
variables VARS. Return the resulting polynomial or list of
polynomials.  Standard arithmetical operators in form EXPR are
replaced with their analogues in the ring of polynomials, and the
resulting expression is evaluated, resulting in a polynomial or a list
of polynomials in internal form. A similar operation in another computer
algebra system could be called 'expand' or so."
  (declare (type ring ring))
  (labels ((p-eval (arg) (poly-eval arg vars ring order))
	   (p-eval-scalar (arg) (poly-eval-scalar arg))
	   (p-eval-list (args) (mapcar #'p-eval args))
	   (p-add (x y) (poly-add ring-and-order x y)))
    (cond
      ((null expr) (error "Empty expression"))
      ((eql expr 0) (make-poly-zero))
      ((member expr vars :test #'equalp)
       (let ((pos (position expr vars :test #'equalp)))
	 (make-poly-variable ring (length vars) pos)))
      ((atom expr)
       (coerce-coeff ring expr vars))
      ((eq (car expr) list-marker)
       (cons list-marker (p-eval-list (cdr expr))))
      (t
       (case (car expr)
	 (+ (reduce #'p-add (p-eval-list (cdr expr))))
	 (- (case (length expr)
	      (1 (make-poly-zero))
	      (2 (poly-uminus ring (p-eval (cadr expr))))
	      (3 (poly-sub ring-and-order (p-eval (cadr expr)) (p-eval (caddr expr))))
	      (otherwise (poly-sub ring-and-order (p-eval (cadr expr))
				   (reduce #'p-add (p-eval-list (cddr expr)))))))
	 (*
	  (if (endp (cddr expr))		;unary
	      (p-eval (cdr expr))
	      (reduce #'(lambda (p q) (poly-mul ring-and-order p q)) (p-eval-list (cdr expr)))))
	 (/ 
	  ;; A polynomial can be divided by a scalar
	  (cond 
	    ((endp (cddr expr))
	     ;; A special case (/ ?), the inverse
	     (coerce-coeff ring (apply (ring-div ring) (cdr expr)) vars))
	    (t
	     (let ((num (p-eval (cadr expr)))
		   (denom-inverse (apply (ring-div ring)
					 (cons (funcall (ring-unit ring)) 
					       (mapcar #'p-eval-scalar (cddr expr))))))
	       (scalar-times-poly ring denom-inverse num)))))
	 (expt
	  (cond
	    ((member (cadr expr) vars :test #'equalp)
	     ;;Special handling of (expt var pow)
	     (let ((pos (position (cadr expr) vars :test #'equalp)))
	       (make-poly-variable ring (length vars) pos (caddr expr))))
	    ((not (and (integerp (caddr expr)) (plusp (caddr expr))))
	     ;; Negative power means division in coefficient ring
	     ;; Non-integer power means non-polynomial coefficient
	     (coerce-coeff ring expr vars))
	    (t (poly-expt ring-and-order (p-eval (cadr expr)) (caddr expr)))))
	 (otherwise
	  (coerce-coeff ring expr vars)))))))

(defun poly-eval-scalar (expr 
			 &optional
			   (ring +ring-of-integers+)
			 &aux 
			   (order #'lex>))
  "Evaluate a scalar expression EXPR in ring RING."
  (declare (type ring ring))
  (poly-lc (poly-eval expr nil ring order)))

(defun spoly (ring-and-order f g
	      &aux
		(ring (ro-ring ring-and-order)))
  "It yields the S-polynomial of polynomials F and G."
  (declare (type ring-and-order ring-and-order) (type poly f g))
  (let* ((lcm (monom-lcm (poly-lm f) (poly-lm g)))
	  (mf (monom-div lcm (poly-lm f)))
	  (mg (monom-div lcm (poly-lm g))))
    (declare (type monom mf mg))
    (multiple-value-bind (c cf cg)
	(funcall (ring-ezgcd ring) (poly-lc f) (poly-lc g))
      (declare (ignore c))
      (poly-sub 
       ring-and-order
       (scalar-times-poly ring cg (monom-times-poly mf f))
       (scalar-times-poly ring cf (monom-times-poly mg g))))))


(defun poly-primitive-part (ring p)
  "Divide polynomial P with integer coefficients by gcd of its
coefficients and return the result."
  (declare (type ring ring) (type poly p))
  (if (poly-zerop p)
      (values p 1)
    (let ((c (poly-content ring p)))
      (values (make-poly-from-termlist 
	       (mapcar
		#'(lambda (x)
		    (make-term :monom (term-monom x)
			       :coeff (funcall (ring-div ring) (term-coeff x) c)))
		(poly-termlist p))
	       (poly-sugar p))
	      c))))

(defun poly-content (ring p)
  "Greatest common divisor of the coefficients of the polynomial P. Use the RING structure
to compute the greatest common divisor."
  (declare (type ring ring) (type poly p))
  (reduce (ring-gcd ring) (mapcar #'term-coeff (rest (poly-termlist p))) :initial-value (poly-lc p)))

(defun read-infix-form (&key (stream t))
  "Parser of infix expressions with integer/rational coefficients
The parser will recognize two kinds of polynomial expressions:

- polynomials in fully expanded forms with coefficients
  written in front of symbolic expressions; constants can be optionally
  enclosed in (); for example, the infix form
  	X^2-Y^2+(-4/3)*U^2*W^3-5
  parses to
	(+ (- (EXPT X 2) (EXPT Y 2)) (* (- (/ 4 3)) (EXPT U 2) (EXPT W 3)) (- 5))

- lists of polynomials; for example
        [X-Y, X^2+3*Z]          
  parses to
	  (:[ (- X Y) (+ (EXPT X 2) (* 3 Z)))
  where the first symbol [ marks a list of polynomials.

-other infix expressions, for example
        [(X-Y)*(X+Y)/Z,(X+1)^2]
parses to:
	(:[ (/ (* (- X Y) (+ X Y)) Z) (EXPT (+ X 1) 2))
Currently this function is implemented using M. Kantrowitz's INFIX package."
  (read-from-string
   (concatenate 'string
     "#I(" 
     (with-output-to-string (s)
       (loop
	 (multiple-value-bind (line eof)
	     (read-line stream t)
	   (format s "~A" line)
	   (when eof (return)))))
     ")")))
	
(defun read-poly (vars &key
			 (stream t) 
			 (ring +ring-of-integers+) 
			 (order #'lex>))
  "Reads an expression in prefix form from a stream STREAM.
The expression read from the strem should represent a polynomial or a
list of polynomials in variables VARS, over the ring RING.  The
polynomial or list of polynomials is returned, with terms in each
polynomial ordered according to monomial order ORDER."
  (poly-eval (read-infix-form :stream stream) vars ring order))

(defun string->poly (str vars 
		     &optional
		       (ring +ring-of-integers+) 
		       (order #'lex>))
  "Converts a string STR to a polynomial in variables VARS."
  (with-input-from-string (s str)
    (read-poly vars :stream s :ring ring :order order)))

(defun poly->alist (p)
  "Convert a polynomial P to an association list. Thus, the format of the
returned value is  ((MONOM[0] . COEFF[0]) (MONOM[1] . COEFF[1]) ...), where
MONOM[I] is a list of exponents in the monomial and COEFF[I] is the
corresponding coefficient in the ring."
  (cond
    ((poly-p p)
     (mapcar #'term->cons (poly-termlist p)))
    ((and (consp p) (eq (car p) :[))
     (cons :[ (mapcar #'poly->alist (cdr p))))))

(defun string->alist (str vars
		     &optional
		       (ring +ring-of-integers+) 
		       (order #'lex>))
  "Convert a string STR representing a polynomial or polynomial list to
an association list (... (MONOM . COEFF) ...)."
  (poly->alist (string->poly str vars ring order)))

(defun poly-equal-no-sugar-p (p q)
  "Compare polynomials for equality, ignoring sugar."
  (declare (type poly p q))
  (equalp (poly-termlist p) (poly-termlist q)))

(defun poly-set-equal-no-sugar-p (p q)
  "Compare polynomial sets P and Q for equality, ignoring sugar."
  (null (set-exclusive-or  p q :test #'poly-equal-no-sugar-p )))

(defun poly-list-equal-no-sugar-p (p q)
  "Compare polynomial lists P and Q for equality, ignoring sugar."
  (every #'poly-equal-no-sugar-p p q))
|#
